/******************************************************************************
 ** $Id: Kombinatorik.java 899 2016-02-21 17:25:15Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 6.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

/** Sammlung von allgemeinen Formeln zum Thema Kombinatorik. */
public final class Kombinatorik {

    /** Anzahl Mglichkeiten, wenn alle Wrfel geworfen werden. */
    public static final int VARIATIONEN = 7776;
    /** Wahrscheinlichkeit bei einem Wrfel bei einem Wurf eine bestimmte Augenzahl auszulassen. */
    public static final double AUSLASSUNGSWAHRSCHEINLICHKEIT = 5. / 6.;

    /** Mittlere Augenzahl bei 0, 1, 2 oder 3 Restwrfen. */
    private static final double[] AUGENZAHLEN = { 0, 3.5, 4.25, 14. / 3. };
    /** Wahrscheinlichkeit fr 1, ..., 5 gleiche Augen in einem Wurf mit allen Wrfeln. */
    private static final double[] GLEICHENWAHRSCHEINLICHKEIT = { 5. / 54., 25. / 36., 125. / 648., 25. / 1296.,
            1 / 1296. };
    /** Stirling-Zahlen der zweiten Art. S(n,k) = k * S(n-1,k) + S(n-1,k-1); S(n,0) = 0, S(n,1) = 1 */
    private static final int[][] STIRLING2;

    static {
        STIRLING2 = new int[Turnier.WUERFEL + 1][];
        STIRLING2[0] = new int[1];
        STIRLING2[0][0] = 1;
        for (int n = 1; n <= Turnier.WUERFEL; n++) {
            STIRLING2[n] = new int[n + 1];
            STIRLING2[n][0] = 0;
            for (int k = 1; k < n; k++) {
                STIRLING2[n][k] = k * STIRLING2[n - 1][k] + STIRLING2[n - 1][k - 1];
            }
            STIRLING2[n][n] = 1;
        }
    }

    /**
     * @param rest
     *            Restliche Anzahl der freien Wrfe.
     * @return Mittlere erzielbare Augenzahl des Wrfels.
     */
    public static double augenzahl(int rest) {
        return AUGENZAHLEN[rest];
    }

    /**
     * @param n
     *            Nichtnegative Zahl.
     * @param k
     *            Ganze Zahl.
     * @return Binomialkoeffizient n ber k.
     */
    public static int binom(int n, int k) {
        if (k < 0 || k > n) {
            return 0;
        }
        if (2 * k > n) {
            return binom(n, n - k);
        }
        int koeffizient = 1;
        n -= k;
        for (int i = 1; i <= k; i++) {
            n++;
            koeffizient *= n;
            koeffizient /= i;
        }
        return koeffizient;
    }

    /**
     * @param n
     *            Nichtnegative Zahl
     * @return Fakultt
     */
    public static int fakultaet(int n) {
        int p = 1;
        while (n > 0) {
            p *= n;
            n--;
        }
        return p;
    }

    /**
     * @param n
     *            Startfaktor.
     * @param k
     *            Anzahl der Faktoren.
     * @return Fallende Fakultt.
     */
    public static int fallend(int n, int k) {
        int p;
        if (k < 0) {
            p = 0;
        } else {
            p = 1;
            for (int i = 0; i < k; i++, n--) {
                p *= n;
            }
        }
        return p;
    }

    /**
     * @param n
     *            Anzahl der Wrfel.
     * @param r
     *            Anzahl der Restwrfe.
     * @return Mittlere Anzahl gleicher Augenzahlen wie eine fest vorgegebene Augenzahl.
     */
    public static double gleichenmittel(int n, int r) {
        double p = 1 - Math.pow(AUSLASSUNGSWAHRSCHEINLICHKEIT, r);
        return n * p;
    }

    /**
     * @param m
     *            Anzahl der Wrfel mit gleicher Augenzahl (1, 2, 3, 4, 5).
     * @return Wahrscheinlichkeit fr genau m Wrfel mit gleicher Augenzahl.
     */
    public static double gleichenwahrscheinlichkeit(int m) {
        return GLEICHENWAHRSCHEINLICHKEIT[m - 1];
    }

    /**
     * @param n
     *            Anzahl der Wrfel.
     * @param r
     *            Anzahl der Restwrfe (0 ist erlaubt).
     *
     * @return Wahrscheinlichkeit fr genau n gleiche Wrfel bei festgelegter Augenzahl.
     */
    public static double gleichenwahrscheinlichkeit(int n, int r) {
        double p = 1 - Math.pow(AUSLASSUNGSWAHRSCHEINLICHKEIT, r);
        return Math.pow(p, n);
    }

    /**
     * @param n
     *            Anzahl der Wrfel.
     * @param k
     *            Anzahl der gleichen Wrfel bei festgelegter Augenzahl.
     * @param r
     *            Anzahl der Restwrfe.
     *
     * @return Wahrscheinlichkeit fr genau k gleiche Wrfel bei festgelegter Augenzahl.
     */
    public static double gleichenwahrscheinlichkeit(int n, int k, int r) {
        double p = 1 - Math.pow(AUSLASSUNGSWAHRSCHEINLICHKEIT, r);
        return binom(n, k) * Math.pow(p, k) * Math.pow(1 - p, n - k);
    }

    /**
     * @param min
     *            Mindestanzahl der Wrfel, welche eine bestimmte Augenzahl zeigen sollen.
     * @return Wahrscheinlichkeit fr die Mindestanzahl.
     */
    public static double mindestgleichenwahrscheinlichkeit(int min) {
        return mindestgleichenwahrscheinlichkeit(Turnier.WUERFEL, min, Turnier.RUNDEN);
    }

    /**
     * @param n
     *            Anzahl der Wrfel, die zur Verfgung stehen.
     * @param min
     *            Mindestanzahl der Wrfel, welche eine bestimmte Augenzahl zeigen sollen.
     * @param r
     *            Anzahl der verbleibenden Restwrfe.
     * @return Wahrscheinlichkeit fr die Mindestanzahl.
     */
    public static double mindestgleichenwahrscheinlichkeit(int n, int min, int r) {
        double p = 0;
        for (int k = min; k <= Turnier.WUERFEL; k++) {
            p += gleichenwahrscheinlichkeit(n, k, r);
        }
        return p;
    }

    /**
     * Berechnet Stirlingzahl der zweiten Art (Anzahl der Partitionen der Menge der Kardinalitt n in k nicht-leere
     * Mengen).
     *
     * @param n
     *            Kardinalitt der Menge.
     * @param k
     *            Anzahl der nicht-leeren Mengen.
     * @return Stirlingzahl der zweiten Art.
     */
    public static int stirling2(int n, int k) {
        if (k <= n) {
            return STIRLING2[n][k];
        } else {
            return 0;
        }
    }

    /**
     * @param n
     * @param d
     * @param k
     * @return
     */
    public static int strassenteilschablone(int n, int d, int k) {
        int summe = 0;
        for (int i = k; i <= n; i++) {
            summe += stirling2(i, k) * binom(n, i) * Math.pow(d, n - i);
        }
        return summe;
    }

    /**
     * @param n
     * @param d
     * @param k
     * @param m
     * @return
     */
    public static double strassenteilsequenz(int n, int d, int k, int m) {
        return fallend(m, k) * strassenteilschablone(n, d, k);
    }

    /** Diese Klasse liefert nur statische Methode, daher wird eine Instanziierung verboten. */
    private Kombinatorik() {
    }

}
