/******************************************************************************
 ** $Id: JaFuffy.java 1017 2016-05-28 20:35:09Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy;

import jafuffy.bedienung.Ende;
import jafuffy.bedienung.Fenster;
import jafuffy.bedienung.Menue;
import jafuffy.bedienung.Oberflaeche;
import jafuffy.bedienung.Signal;
import jafuffy.bedienung.Spielplatz;
import jafuffy.bedienung.Willkommen;
import jafuffy.logik.Manager;
import jafuffy.logik.Statistik;
import jafuffy.logik.Update;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.RandomAccessFile;
import java.net.URISyntaxException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;

import javax.swing.JApplet;
import javax.swing.JOptionPane;
import javax.swing.border.EmptyBorder;

/** JaFuffy (Hauptteil, Applet oder Applikation). */
@SuppressWarnings("serial")
public class JaFuffy extends JApplet {

    /** Mindestversion Java. */
    private static final String MINDEST_LANG_VERSION = "1.7";
    /** Mindestversion Swing. */
    private static final String MINDEST_SWING_VERSION = "1.7";
    /** Abstand vom Fensterrahmen zum Inhalt. */
    private static final int RAHMENABSTAND = 4;
    /** Lock-Datei zur Vermeidung eines Mehrfachstarts. */
    private static final String VERRIEGELUNGSDATEI = "JaFuffy.lock";

    /**
     * Einstieg fr die Applikation.
     *
     * @param args
     *            Kommandozeilenargumente.
     */
    public static void main(String[] args) {
        Oberflaeche.starteJetzt(new Runnable() {

            @Override
            public void run() {
                pruefeKompatibilitaet();
            }
        });
        werteArgumenteAus(args);
        try {
            if (laeuftSchon()) {
                JaFuffyDialoge.meldeMehrfachstart();
                System.exit(0);
            }
        } catch (IOException exception) {
            // Probleme mit Mehrfachstartdetektion melden und trotzdem weitermachen
            exception.printStackTrace();
        }
        Eigenschaften eigenschaften = Eigenschaften.GLOBAL;
        if (eigenschaften.erststart()) {
            new Willkommen(eigenschaften);
        }
        registriereAufraeumer(eigenschaften);
        new JaFuffy(eigenschaften, beschaffeStatistik());
    }

    /**
     * Liest die Statistik-Daten fr die Applikation ein oder aber erzeugt diese neu.
     *
     * @return Statistik-Objekt.
     */
    private static Statistik beschaffeStatistik() {
        File datei = Eigenschaften.GLOBAL.datei(Statistik.STAT);
        if (datei.exists()) {
            Statistik statistik;
            try {
                ObjectInputStream ois = new ObjectInputStream(new FileInputStream(datei));
                statistik = (Statistik) ois.readObject();
                ois.close();
                return statistik;
            } catch (Exception e) {
                e.printStackTrace();
                Signal.ALARM.play();
                JaFuffyDialoge.meldeFehlendeStatistikdaten(e);
            }
        }
        return new Statistik();
    }

    /**
     * Prft auf eine Lock-Datei und damit ob JaFuffy schon als Anwendung gestartet wurde, und registriert eine
     * Aufrumfunktion fr die Lock-Datei.
     *
     * @return Ist JaFuffy schon gestartet worden?
     *
     * @throws IOException
     *             Probleme mit der Lock-Datei sind aufgetreten.
     */
    private static boolean laeuftSchon() throws IOException {
        final File datei = new File(System.getProperty("java.io.tmpdir"), VERRIEGELUNGSDATEI);
        final RandomAccessFile stream = new RandomAccessFile(datei, "rw");
        final FileChannel kanal = stream.getChannel();
        final FileLock riegel = kanal.tryLock();
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                try {
                    stream.close();
                } catch (IOException exception) {
                    exception.printStackTrace();
                }
                datei.delete();
            }
        });
        return riegel == null;
    }

    /**
     * Prfe auf die Kompatibilitt mit Java-Version und Swing-Version und zeige gegebenenfalls einen Warnhinweis.
     */
    private static void pruefeKompatibilitaet() {
        Package lang = Package.getPackage("java.lang");
        Package swing = Package.getPackage("javax.swing");
        if (!lang.isCompatibleWith(MINDEST_LANG_VERSION) || !swing.isCompatibleWith(MINDEST_SWING_VERSION)) {
            JaFuffyDialoge.meldeVersionsproblem(lang, MINDEST_LANG_VERSION, swing, MINDEST_SWING_VERSION);
        }
    }

    /**
     * Registriert Aufrumthread, der bei Programmende aufgerufen wird und gegebenenfalls einen Neustart durchfhrt.
     *
     * @param eigenschaften
     *            Die Eigenschaften, mit denen JaFuffy betrieben wird. Darber kann ermittelt werden, ob ein Neustart
     *            vorgenommen werden soll.
     */
    private static void registriereAufraeumer(final Eigenschaften eigenschaften) {
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                if (eigenschaften.neustart()) {
                    try {
                        File jar = new File(JaFuffy.class.getProtectionDomain().getCodeSource().getLocation().toURI());
                        try {
                            String java = System.getProperty("java.home") + "/bin/java";
                            StringBuffer cmd = new StringBuffer("\"" + java + "\"");
                            cmd.append(" -jar \"").append(jar.getPath()).append("\"");
                            Runtime.getRuntime().exec(cmd.toString());
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                    } catch (URISyntaxException exception) {
                        exception.printStackTrace();
                    }
                }
            }
        });
    }

    /**
     * Wertet die Kommandozeilenargumente aus.
     *
     * @param args
     *            Kommandozeilenargumente.
     */
    private static void werteArgumenteAus(String[] args) {
        if (args.length > 1) {
            System.err.println("Maximal ein Argument erlaubt!");
        }
        if (args.length == 1) {
            if (args[0].equals("nosound")) {
                Ressource.schalteStumm();
                System.out.println("Sound ist abgeschaltet");
            } else {
                System.err.println("Option nicht bekannt!");
            }
        }
    }

    /**
     * JaFuffy ist als Applet gestartet worden.
     */
    public JaFuffy() {
    }

    /**
     * JaFuffy ist als Applikation gestartet worden.
     *
     * @param eigenschaften
     *            Die Eigenschaften, mit denen JaFuffy betrieben wird.
     * @param statistik
     *            Statistik, mit der JaFuffy betrieben wird.
     */
    public JaFuffy(final Eigenschaften eigenschaften, final Statistik statistik) {
        Oberflaeche.starteSpaeter(new Runnable() {

            @Override
            public void run() {
                Fenster fenster = new Fenster("JaFuffy");
                Ende ende = new Ende(eigenschaften, statistik);
                fenster.addWindowListener(ende);
                Update update = new Update();
                new Manager(eigenschaften, statistik, fenster, ende, update);
                update.ueberpruefe(eigenschaften.aktiv("Nachschauen"), fenster);

            }
        });
    }

    /** JaFuffy ist als Applet gestartet worden, hier wird es initialisiert. */
    @Override
    public final void init() {
        Oberflaeche.starteJetzt(new Runnable() {

            @Override
            public void run() {
                pruefeKompatibilitaet();
                new Manager(Eigenschaften.GLOBAL, new Statistik(), JaFuffy.this);
            }
        });
    }

    /**
     * Zeit Men und Spielplatz im Applet-Feld.
     *
     * @param menue
     *            Men.
     * @param spielplatz
     *            Spielplatz.
     */
    public void zeige(Menue menue, Spielplatz spielplatz) {
        setJMenuBar(menue);
        spielplatz.setBorder(new EmptyBorder(RAHMENABSTAND, RAHMENABSTAND, RAHMENABSTAND, RAHMENABSTAND));
        setContentPane(spielplatz);
    }

}

/** Von der Klasse JaFuffy bentigte Dialoge. */
@SuppressWarnings("serial")
class JaFuffyDialoge extends JOptionPane {

    /**
     * Fehlerhinweis falls Statistikdaten nicht geladen werden knnen.
     *
     * @param exception
     *            Exception, die beim Ladeversuch aufgetreten ist.
     */
    static void meldeFehlendeStatistikdaten(Exception exception) {
        showMessageDialog(null, "Statistikdaten knnen nicht geladen werden!\n"
                + "Datenverlust bei nchster Sicherung mglich.\n" + "Java-Exception:\n" + exception,
                "JaFuffy (Hinweis)", INFORMATION_MESSAGE);
    }

    /** Fehlerhinweis falls JaFuffy mehrfach gestartet wurde. */
    static void meldeMehrfachstart() {
        showMessageDialog(null, "JaFuffy wurde offenbar schon gestartet!\n" + "Diese Instanz beendet sich jetzt.",
                "JaFuffy (Hinweis)", INFORMATION_MESSAGE);
    }

    /**
     * Hinweis auf Versionsproblem.
     *
     * @param lang
     *            Gefundene Javasprachversion.
     * @param minLang
     *            Minimale Javasprachversion.
     * @param swing
     *            Gefundene Swingversion.
     * @param minSwing
     *            Minimale Swingversion.
     */
    static void meldeVersionsproblem(Package lang, String minLang, Package swing, String minSwing) {
        JOptionPane.showMessageDialog(null,
                "Versionsforderungen an Java-Pakete nicht erfllt!\n" + "Mglicherweise treten Probleme auf.\n\n"
                        + "Paket \"" + lang + "\" gefunden,\n" + "es wird mindestens Version " + minLang
                        + " bentigt.\n" + "Paket \"" + swing + "\" gefunden\n" + "es wird mindestens Version "
                        + minSwing + " bentigt.",
                "JaFuffy" + " (Warnung)", WARNING_MESSAGE);
    }

}
