/******************************************************************************
 ** $Id: Verlaufsgrafik.java 1017 2016-05-28 20:35:09Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.GradientPaint;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;

import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JViewport;
import javax.swing.SwingUtilities;

import jafuffy.logik.Spieler;
import jafuffy.logik.Statistik;

/** Anzeige des Turnierverlaufs, nach jeder Runde aktualisiert. */
@SuppressWarnings("serial")
class Verlaufsgrafik extends JPanel {

    /** Der Rand zum Zeichenfeld, welcher freigelassen wird. */
    private static final int RAND = 20;
    /**
     * Basisgre zur Bestimmung der Vergrerung des Zeichenfeldes, welche grob angibt, wie viele Spiele im
     * Sichtfenster zu sehen sind.
     */
    private static final int FAKTORBASIS = 10;
    /** Steuert den Abstand der Beschriftung zu den Balken. */
    private static final int BESCHRIFTUNG = 20;
    // Konstanten fr Balkengrafik (Abstnde)
    private static final double LUECKE_HOR = 50;
    private static final double LUECKE_VERT = 15;
    private static final double BALKENBREITE = 75;
    private static final double BALKENTIEFE = 15;
    // Neigung fr Perspektive
    private static final double STEIGUNG = 0.5;
    // Hilfslinie
    private static final float[] DASH = { 16 };
    // Farbe
    private static final Color WAND_RECHTS = new Color(240, 240, 240);
    private static final Color WAND_UNTEN = new Color(235, 235, 235);
    private static final Color WAND_HINTEN = new Color(245, 245, 245);
    private static final Color[] BALKEN_UNTEN = new Color[Spieler.SPIELER];
    private static final Color[] BALKEN_OBEN = new Color[Spieler.SPIELER];

    // Farbverlauf unten -> oben
    static {
        for (int i = 0; i < Spieler.SPIELER; i++) {
            BALKEN_UNTEN[i] = Report.FARBEN[i].darker();
            BALKEN_OBEN[i] = Report.FARBEN[i].brighter();
        }
    }

    // Zoomfunktionen
    private final JPopupMenu popup = new JPopupMenu();
    private final JMenuItem kleiner = new JMenuItem("Verkleinern");
    private final JMenuItem groesser = new JMenuItem("Vergrern");
    private final JMenuItem normal = new JMenuItem("Standard");
    private final JMenuItem aktuell = new JMenuItem("Aktuell");

    // statistische Daten & Spielverlauf
    private final Statistik statistik;
    private ArrayList<Integer>[] verlauf;
    // Anzahl der Spiele (tatschlich gespielt bzw. Vorgabe pro Turnier)
    private int anzahl;
    // max. Hhe fr Balkengrafik (ber Null)
    private double maximalhoehe;
    // Hhe der hinteren Koordinatenwand
    private double wandhoehe;
    /** Transformation zum Zeichnen des Balkendiagramms. */
    private AffineTransform trafo;

    /** Zoom-Faktor zur Vergrerung des Zeichenfeldes fr die Balkengrafik. */
    private int faktor = 1;

    /**
     * Konstruktor.
     *
     * @param statistik
     *            Die Statistik, die grafisch als Balkendiagramm dargestellt wird.
     */
    Verlaufsgrafik(Statistik statistik) {

        setBackground(Color.white);

        this.statistik = statistik;

        setToolTipText("<html><p>Aufschlsselung nach den einzelnen Spielen.</p>"
                + "<p>Zoomfunktionen auf rechter Maustaste.</p>" + "<p>Ziehen (\"Dragging\") verfgbar.</p></html>");

        // kleiner/grer/normal
        ActionListener zoom = new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent event) {
                zoome(event);
            }
        };
        kleiner.setEnabled(false);
        kleiner.addActionListener(zoom);
        groesser.addActionListener(zoom);
        normal.setEnabled(false);
        normal.addActionListener(zoom);
        aktuell.addActionListener(zoom);
        aktuell.setToolTipText("Letzten Stand anzeigen");
        // Popup-Men
        popup.add(kleiner);
        popup.add(groesser);
        popup.addSeparator();
        popup.add(normal);
        popup.addSeparator();
        popup.add(aktuell);
        addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                maybeShowPopup(e);
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                maybeShowPopup(e);
            }

            private void maybeShowPopup(MouseEvent e) {
                if (e.isPopupTrigger()) {
                    kleiner.setEnabled(faktor > 1);
                    normal.setEnabled(faktor > 1);
                    popup.show(e.getComponent(), e.getX(), e.getY());
                }
            }
        });

    }

    @Override
    public void setVisible(boolean sichtbar) {
        aktualisiere();
    }

    /**
     * Look&Feel korrekt ndern (auch Kontextmen).
     */
    @Override
    public void updateUI() {
        super.updateUI();
        if (popup != null) {
            SwingUtilities.updateComponentTreeUI(popup);
        }
    }

    /** Bestimmt einen heuristisch ermittelten Vergrerungsfaktor. */
    private void bestimmeFaktor() {
        faktor = (int) Math
                .round(Math.pow(2, Math.floor(Math.log(Math.ceil(anzahl / (double) FAKTORBASIS)) / Math.log(2))));
    }

    /**
     * @param s
     *            Nummer des Spielers
     * @param punkte
     *            Balkenhhe
     * @return Farbe des Balkendeckels.
     */
    private Color deckelfarbe(int s, int punkte) {

        int r1 = BALKEN_UNTEN[s].getRed();
        int g1 = BALKEN_UNTEN[s].getGreen();
        int b1 = BALKEN_UNTEN[s].getBlue();
        int r2 = BALKEN_OBEN[s].getRed();
        int g2 = BALKEN_OBEN[s].getGreen();
        int b2 = BALKEN_OBEN[s].getBlue();

        if (punkte < wandhoehe) {
            return new Color(r1 + (r2 - r1) * punkte / (int) wandhoehe, g1 + (g2 - g1) * punkte / (int) wandhoehe,
                    b1 + (b2 - b1) * punkte / (int) wandhoehe);
        } else {
            return BALKEN_OBEN[s];
        }

    }

    /**
     * Erstellt Transformation und zeichnet hintere, seitliche und untere Koordinatenwnde.
     *
     * @param g2
     *            Grafikumfeld
     * @param zeichnen
     *            Gibt an, ob neben der Transformation auch die Koordinatenwnde samt Beschriftung gezeichnet werden
     *            sollen.
     */
    private void erstelleKoordinaten(Graphics2D g2, boolean zeichnen) {

        super.paintComponent(g2);
        g2.setFont(new Font("Dialog", Font.PLAIN, BESCHRIFTUNG));

        FontMetrics fm = g2.getFontMetrics();
        double beschriftungslaenge = BESCHRIFTUNG / 2 + fm.stringWidth(String.valueOf(statistik.mittelwert()));
        double nutzbreite = getWidth() - beschriftungslaenge - 2 * RAND;
        double nutzhoehe = getHeight() - 2 * RAND;
        double grundbreite = statistik.spieler().size() * (BALKENBREITE + LUECKE_HOR) - LUECKE_HOR;
        double grundhoehe = -y(anzahl);
        double breite = grundbreite + grundhoehe / STEIGUNG;
        double hoehe = grundhoehe + maximalhoehe;
        double skalierung;

        // merke Ursprung
        trafo = g2.getTransform();

        // affine Transformation fr die Umrechnung aus abstrakten
        // 2D-Koordinaten in Bildschirmdarstellung. Fallunterscheidung
        // langes/hohes Grafikfeld.
        if (hoehe / breite <= nutzhoehe / nutzbreite) {
            skalierung = nutzbreite / breite;
            g2.translate(grundbreite * skalierung + RAND,
                    (nutzhoehe - hoehe * skalierung) / 2 + maximalhoehe * skalierung + RAND);
        } else {
            skalierung = nutzhoehe / hoehe;
            g2.translate((nutzbreite - breite * skalierung) / 2 + grundbreite * skalierung + RAND,
                    maximalhoehe * skalierung + RAND);
        }
        g2.scale(skalierung, skalierung);

        // Beschriftung
        if (zeichnen) {
            for (int punkte = 0; punkte <= wandhoehe; punkte += 100) {
                g2.drawString(String.valueOf(punkte), (float) (BESCHRIFTUNG / 2 + grundhoehe / STEIGUNG),
                        (float) (grundhoehe - punkte - BESCHRIFTUNG / 4));
            }
        }

        // endgltige Transformation auf Bildschirm
        g2.scale(-1, -1);
        trafo = g2.getTransform();

        // Nur Trafo verlangt?
        if (!zeichnen) {
            return;
        }

        // Grafikgren
        Rectangle2D rechteck;
        BasicStroke dick = new BasicStroke((float) (2 / skalierung));
        BasicStroke normal = new BasicStroke((float) (1 / skalierung));
        BasicStroke gestrichelt =
                new BasicStroke((float) (1 / skalierung), BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL, 1, DASH, 0);

        // hintere Wand
        rechteck = new Rectangle2D.Double(0, 0, grundbreite, wandhoehe);
        g2.setTransform(trafo);
        g2.setPaint(WAND_HINTEN);
        g2.fill(rechteck);
        g2.setPaint(Color.black);
        g2.setStroke(dick);
        g2.draw(rechteck);
        g2.setStroke(normal);
        for (double y = 100; y < wandhoehe; y += 100) {
            g2.draw(new Line2D.Double(0, y, grundbreite, y));
        }
        g2.setStroke(gestrichelt);
        for (double y = 50; y < wandhoehe; y += 100) {
            g2.draw(new Line2D.Double(0, y, grundbreite, y));
        }

        // rechte Wand
        rechteck = new Rectangle2D.Double(-grundhoehe / STEIGUNG, 0, grundhoehe / STEIGUNG, wandhoehe);
        g2.setTransform(trafo);
        g2.shear(0, STEIGUNG);
        g2.setPaint(WAND_RECHTS);
        g2.fill(rechteck);
        g2.setPaint(Color.black);
        g2.setStroke(dick);
        g2.draw(rechteck);
        g2.setStroke(normal);
        for (double y = 100; y < wandhoehe; y += 100) {
            g2.draw(new Line2D.Double(-grundhoehe / STEIGUNG, y, 0, y));
        }
        g2.setStroke(gestrichelt);
        for (double y = 50; y < wandhoehe; y += 100) {
            g2.draw(new Line2D.Double(-grundhoehe / STEIGUNG, y, 0, y));
        }

        // Boden
        rechteck = new Rectangle2D.Double(0, -grundhoehe, grundbreite, grundhoehe);
        g2.setTransform(trafo);
        g2.shear(1 / STEIGUNG, 0);
        g2.setPaint(WAND_UNTEN);
        g2.fill(rechteck);
        g2.setPaint(Color.black);
        g2.setStroke(dick);
        g2.draw(rechteck);

        // Linienstil zurcksetzen auf mittel
        g2.setStroke(new BasicStroke((float) (1.5 / skalierung)));

    }

    /**
     * @param s
     *            Nummer des Spielers
     * @param i
     *            Nummer des Spiels im Turnier
     * @return x-Wert des rechten hinteren Balkenfupunktes
     */
    private double x(int s, int i) {
        return (BALKENBREITE + LUECKE_HOR) * (statistik.spieler().size() - s - 1) + y(i) / STEIGUNG;
    }

    /**
     * @param i
     *            Nummer des Spiels im Turnier
     * @return y-Wert des rechten hinteren Balkenfupunktes
     */
    private double y(int i) {
        return -(BALKENTIEFE + LUECKE_VERT) * i;
    }

    /**
     * Zeichnet Balkengrafik.
     *
     * @param g2
     *            Grafikumfeld
     */
    private void zeichneBalkengrafik(Graphics2D g2) {

        int punkte;
        Rectangle2D rechteck;
        GradientPaint farbverlauf;
        double x;
        double y;

        for (int s = verlauf.length - 1; s >= 0; s--) {
            for (int i = 0; i < verlauf[s].size(); i++) {
                punkte = verlauf[s].get(i).intValue();
                x = x(s, i) - BALKENTIEFE / STEIGUNG; // vorne links unten
                y = y(i) - BALKENTIEFE; // vorne links unten
                if (punkte > 0) {
                    if (y(i) + punkte > maximalhoehe) {
                        maximalhoehe = y(i) + punkte;
                    }
                    // vorne
                    rechteck = new Rectangle2D.Double(x, y, BALKENBREITE, punkte);
                    farbverlauf = new GradientPaint(0, (float) y, BALKEN_UNTEN[s], 0, (float) (y + wandhoehe),
                            BALKEN_OBEN[s]);
                    g2.setPaint(farbverlauf);
                    g2.setTransform(trafo);
                    g2.fill(rechteck);
                    g2.setPaint(Color.black);
                    g2.draw(rechteck);
                    // seitlich
                    rechteck = new Rectangle2D.Double(0, 0, BALKENTIEFE / STEIGUNG, punkte);
                    farbverlauf = new GradientPaint(0, 0, BALKEN_UNTEN[s], 0, (float) wandhoehe, BALKEN_OBEN[s]);
                    g2.setTransform(trafo);
                    g2.translate(x + BALKENBREITE, y);
                    g2.shear(0, STEIGUNG);
                    g2.setPaint(farbverlauf);
                    g2.fill(rechteck);
                    g2.setPaint(Color.black);
                    g2.draw(rechteck);
                    // oben
                    rechteck = new Rectangle2D.Double(0, 0, BALKENBREITE, BALKENTIEFE);
                    g2.setPaint(deckelfarbe(s, punkte));
                    g2.setTransform(trafo);
                    g2.translate(x, y + punkte);
                    g2.shear(1 / STEIGUNG, 0);
                    g2.fill(rechteck);
                    g2.setPaint(Color.black);
                    g2.draw(rechteck);
                }
            }
        }

    }

    /**
     * Vergrerung des Zeichenfeldes festlegen durch Anwendung des Vergrerungsfaktors auf die bevorzugte Gre
     * abgeleitet aus dem sichtbaren Anteil.
     */
    private void zoome() {
        setPreferredSize(new Dimension(faktor * getVisibleRect().width, faktor * getVisibleRect().height));
        revalidate();
    }

    /**
     * ndert die Zeichenfeldgre und rckt bei Bedarf zum aktuellen Stand vor.
     *
     * @param event
     *            Bearbeitet das Ereignis, welches aus dem Kontextmen stammt.
     */
    private void zoome(ActionEvent event) {

        String ac = event.getActionCommand();
        Rectangle r = ((JViewport) getParent()).getViewRect();
        int u = r.x + r.width / 2; // Ausschnittmittelpunkt
        int v = r.y + r.height / 2; // Ausschnittmittelpunkt

        if (ac.equals("Vergrern")) {
            u *= 2;
            v *= 2;
            faktor *= 2;
            zoome();
            scrollRectToVisible(new Rectangle(u - r.width / 2, v - r.height / 2, r.width, r.height));
        } else if (ac.equals("Verkleinern") && faktor > 1) {
            u /= 2;
            v /= 2;
            faktor /= 2;
            zoome();
            scrollRectToVisible(new Rectangle(u - r.width / 2, v - r.height / 2, r.width, r.height));
        } else if (ac.equals("Standard")) {
            faktor = 1;
            zoome();
        } else if (ac.equals("Aktuell")) {
            aktualisiere();
        }

    }

    /**
     * Trafo berechnen, Balkendiagramm neu zeichnen.
     *
     * @param g
     */
    @Override
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        Graphics2D g2 = (Graphics2D) g;
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g2.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);
        erstelleKoordinaten(g2, true);
        zeichneBalkengrafik(g2);
    }

    /** Erstellt das Balkendiagramm zu einem neuen Spielstand und springt an das aktuelle Spielgeschehen. */
    void aktualisiere() {
        erstelleKoordinaten((Graphics2D) getGraphics(), false);
        int aktuell = verlauf[0].size() - 1;
        double[] src = new double[4];
        src[0] = x(0, aktuell) + BALKENBREITE;
        src[1] = y(aktuell) - BALKENTIEFE;
        src[2] = x(verlauf.length - 1, aktuell) - BALKENTIEFE / STEIGUNG;
        src[3] = y(aktuell) + wandhoehe;
        double[] dest = new double[4];
        trafo.transform(src, 0, dest, 0, 2);
        scrollRectToVisible(new Rectangle((int) dest[0] - RAND / 2, (int) dest[3] - RAND / 2,
                (int) (dest[2] - dest[0]) + RAND, (int) (dest[1] - dest[3]) + RAND));
        repaint();
    }

    /** Spiel beendet. */
    void beende() {
        if (statistik.anzahl() == 0) {
            anzahl++;
            bestimmeFaktor();
            zoome();
            SwingUtilities.invokeLater(new Runnable() {
                @Override
                public void run() {
                    aktualisiere();
                }
            });
        }
    }

    /** Neues Turnier gestartet oder Turnier fortgesetzt. */
    void starte() {
        verlauf = statistik.verlauf();
        wandhoehe = Math.ceil((statistik.mittelwert() + statistik.abweichung()) / 100.) * 100.;
        maximalhoehe = wandhoehe;
        anzahl = Integer.max(statistik.anzahl(), verlauf[0].size());
        bestimmeFaktor();
        zoome();
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                aktualisiere();
            }
        });
    }

}
