/******************************************************************************
 ** $Id: Report.java 1198 2017-04-17 14:56:37Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jafuffy.logik.CEJaFuffy;
import jafuffy.logik.CEStatistik;
import jafuffy.logik.Statistik;

/** Report ber den Verlauf des Turniers. */
@SuppressWarnings("serial")
public class Report extends JDialog implements PropertyChangeListener, ChangeListener {

    /** Gewnschte Breite des Reports. */
    private static final int BREITE = 600;
    /** Gewnschte Hhe des Reports. */
    private static final int HOEHE = 580;
    /** Abstand von Elemente zum Dialograhmen. */
    private static final int DIALOGRAHMENABSTAND = 8;
    /** Abstand von Element zu Feldrahmen. */
    private static final int FELDRAHMENABSTAND = 4;
    /** Rahmen fr Hauptgrafiken, also Zwischenstand und Verlauf. */
    private static final Border RAHMEN = BorderFactory.createCompoundBorder(BorderFactory.createRaisedBevelBorder(),
            BorderFactory.createLoweredBevelBorder());

    /** Hintergrundfarbe fr die Visualisierung der Statistik. */
    static final Color HINTERGRUNDFARBE = new Color(250, 252, 250);
    /** Farben fr Balkengrafiken, Index entspricht Spieler im Turnier. */
    static final Color[] BALKENFARBEN =
            { Color.MAGENTA.darker(), Color.GREEN.darker(), Color.ORANGE.darker(), Color.CYAN.darker() };

    /** Die Legende mit dem Farbcode zu den Darstellungen. */
    private final JPanel legende = new JPanel(new GridLayout(0, 1, 0, DIALOGRAHMENABSTAND));
    /** Grafische Darstellung des Turnierverlaufs. */
    private final Verlaufsgrafik verlaufsgrafik;
    /** Tabellarische Darstellung des Turnierverlaufs. */
    private final Verlaufstabelle verlaufstabelle;
    /** Zwischenstand ber den Gesamtpunktstand. */
    private final Zwischenstand zwischenstand;

    /**
     * Erzeugt Dialogfenster fr den Bericht ber den Spielverlauf.
     *
     * @param statistik
     *            Statistische Daten, die hier im Verlauf fr ein Turnier dargestellt werden sollen.
     */
    public Report(Statistik statistik) {

        UIManager.addPropertyChangeListener(this);

        zwischenstand = new Zwischenstand();
        verlaufsgrafik = new Verlaufsgrafik();
        verlaufsgrafik.setBorder(RAHMEN);
        verlaufstabelle = new Verlaufstabelle();

        JScrollPane zwischenstandscrolling = new JScrollPane(zwischenstand);
        zwischenstandscrolling.setBorder(RAHMEN);

        JTabbedPane kartei = new JTabbedPane();
        kartei.addTab("Balkendiagramm zum Verlauf", verlaufsgrafik);
        kartei.addTab("Tabellenauflistung des Verlaufs", verlaufstabellenfeld());
        kartei.setToolTipTextAt(0,
                "<html>" + "<p>Rotieren mittels Maus-Drag.</p>" + "<p>Zoomen mittels Mausrad.</p>"
                        + "<p>Verschieben mittels mittlerer Maustaste oder Maus-Drag mit Alt-Taste.</p>"
                        + "<p>Doppelklick stellt Standardansicht wieder her.</p>" + "</html>");
        kartei.setToolTipTextAt(1, "<html><p>Sortieren durch Anklicken der Spaltenkpfe</p></html>");

        setContentPane(feld(zwischenstandscrolling, kartei));
        setIconImages(Oberflaeche.LOGOS);
        setPreferredSize(new Dimension(BREITE, HOEHE));
        pack();

    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        if (evt.getPropertyName().equals("lookAndFeel")) {
            SwingUtilities.updateComponentTreeUI(this);
        }
    }

    @Override
    public void stateChanged(ChangeEvent ce) {
        Statistik statistik = (Statistik) ce.getSource();
        if (CEJaFuffy.adressiert(ce, CEStatistik.class)) {
            CEJaFuffy<CEStatistik> cej = CEJaFuffy.ereignisbehaelter(ce);
            switch (cej.ereignis()) {
            case TURNIER_START:
            case TURNIER_WEITER:
                starte(statistik);
                zwischenstand.starte(statistik);
                verlaufsgrafik.starte(statistik);
                verlaufstabelle.starte(statistik);
                revalidate();
                break;
            case SPIEL_ENDE:
                verlaufstabelle.beende();
                break;
            case STAND:
                zwischenstand.aktualisiere(statistik);
                verlaufstabelle.aktualisiere();
                break;
            default:
                break;
            }
            setTitle("JaFuffy (Report: " + statistik.titel() + ")");
            repaint();
        }
    }

    /**
     * Baut das Hauptfeld fr den Report zusammen, welche die Verlaufsdarstellungen enthlt.
     *
     * @param zwischenstandscrolling
     *            Stellt den Zwischenstand dar.
     * @param kartei
     *            Enthlt die Darstellungen fr den Verlauf in grafischer und tabellarischer Form.
     * @return Das Hauptfeld.
     */
    private JPanel feld(JScrollPane zwischenstandscrolling, JTabbedPane kartei) {
        JPanel feld = new JPanel(new BorderLayout(0, FELDRAHMENABSTAND));
        feld.add(zwischenstandscrolling, BorderLayout.PAGE_START);
        feld.add(kartei, BorderLayout.CENTER);
        feld.add(leiste(), BorderLayout.PAGE_END);
        feld.setBorder(BorderFactory.createEmptyBorder(DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND, DIALOGRAHMENABSTAND,
                DIALOGRAHMENABSTAND));
        return feld;
    }

    /**
     * Erstellt die Besttigungsleiste fr die Kontrolle durch den Benutzer.
     *
     * @return Leistenfeld.
     */
    private JPanel leiste() {
        JButton schliessen = new JButton("Report schlieen");
        schliessen.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent event) {
                setVisible(false);
            }
        });
        getRootPane().setDefaultButton(schliessen);
        JButton zeigen = new JButton("Legende zeigen");
        zeigen.setToolTipText("Zeigt die Zuordnung von Farben zu Spielern");
        zeigen.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent event) {
                JOptionPane.showMessageDialog(Report.this, legende, "JaFuffy (Legende)",
                        JOptionPane.INFORMATION_MESSAGE);
            }
        });
        JPanel leiste = new JPanel(new GridLayout(1, 2, DIALOGRAHMENABSTAND, 0));
        leiste.add(schliessen);
        leiste.add(zeigen);
        return leiste;
    }

    /**
     * Passt die Legende an ein neues Turnier an.
     *
     * @param statistik
     *            Darzustellende Statistik
     */
    private void starte(Statistik statistik) {
        JLabel name;
        legende.removeAll();
        for (int i = 0; i < statistik.teilnehmer().size(); i++) {
            name = new JLabel(statistik.teilnehmer().get(i).toString(), SwingConstants.CENTER);
            legende.add(name);
            name.setOpaque(true);
            name.setForeground(BALKENFARBEN[i]);
            name.setBackground(Color.white);
            name.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createLineBorder(BALKENFARBEN[i], 2),
                    BorderFactory.createEmptyBorder(FELDRAHMENABSTAND / 2, FELDRAHMENABSTAND, FELDRAHMENABSTAND / 2,
                            FELDRAHMENABSTAND)));
        }
    }

    /**
     * Stellt das Feld (samt Scrolleinrichtung) zusammen, welches die Verlaufstabelle enthlt.
     *
     * @return Verlaufstabellenfeld.
     */
    private JPanel verlaufstabellenfeld() {
        JPanel verlaufstabellenfeld = new JPanel(new FlowLayout());
        verlaufstabellenfeld.setBorder(BorderFactory.createCompoundBorder(RAHMEN, BorderFactory
                .createEmptyBorder(FELDRAHMENABSTAND, FELDRAHMENABSTAND, FELDRAHMENABSTAND, FELDRAHMENABSTAND)));
        verlaufstabellenfeld.setBackground(HINTERGRUNDFARBE);
        JScrollPane verlaufstabellenscrolling = new JScrollPane(verlaufstabelle) {
            @Override
            public Dimension getPreferredSize() {
                Insets rand = getInsets();
                Insets verlaufstabellenfeldrand = verlaufstabellenfeld.getInsets();
                Dimension verlaufstabellengroesse = verlaufstabelle.getPreferredSize();
                Dimension verlaufstabellenfeldgroesse = verlaufstabellenfeld.getSize();
                return new Dimension(
                        verlaufstabellengroesse.width + getVerticalScrollBar().getWidth() + rand.left + rand.right,
                        verlaufstabellenfeldgroesse.height - verlaufstabellenfeldrand.top
                                - verlaufstabellenfeldrand.bottom - rand.top - 1 - rand.bottom - 1);
            }
        };
        verlaufstabellenscrolling.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        verlaufstabellenscrolling.setBorder(BorderFactory.createLoweredBevelBorder());
        verlaufstabellenfeld.add(verlaufstabellenscrolling);
        return verlaufstabellenfeld;
    }

}
