package jafuffy.logik;

import javax.swing.SwingUtilities;

import jafuffy.Eigenschaften;
import jafuffy.bedienung.Absprache;
import jafuffy.bedienung.Becher;
import jafuffy.bedienung.Bestenliste;
import jafuffy.bedienung.Ende;
import jafuffy.bedienung.Fenster;
import jafuffy.bedienung.Fortsetzung;
import jafuffy.bedienung.Kommentierung;
import jafuffy.bedienung.Menue;
import jafuffy.bedienung.Oberflaeche;
import jafuffy.bedienung.Report;
import jafuffy.bedienung.Setzen;
import jafuffy.bedienung.Spielplatz;
import jafuffy.bedienung.Verfolgung;
import jafuffy.bedienung.Vorschlagen;

/** Stellt die Infrastruktur zum Betrieb eines Turniers. */
public class Manager {

    /** Die Absprache zu einem neuen Turnier. */
    private final Absprache absprache;
    /** Sicherheitsfrage zur Besttigung der Programmbeendigung. */
    private final Ende ende;
    /** Fenster, in dem JaFuffy abluft. */
    private final Fenster fenster;
    /** Kmmert sich um die Fortsetzung von gespeicherten Turnieren. */
    private final Fortsetzung fortsetzung;
    /** Men im Fenster. */
    private final Menue menue;
    /** Spielplatz mit Punktezettel und Wrfelfeld. */
    private final Spielplatz spielplatz;
    /** bernimmt die Aufsicht ber die einzelnen Schritte der Bot-Aktionen. */
    private final Aufsicht aufsicht;
    /** Die Statistik, welche fr alle Turnier gefhrt wird. */
    private final Statistik statistik;

    /** Speichert das Turnier, welches zurzeit durch den Manager betrieben wird. */
    private Turnier turnier;
    /**
     * Anzahl der Spieler im letzten Turnier, welche bei Start von JaFuffy der automatischen Sicherung entnommen wird.
     */
    private int vorgaengerspielerzahl;

    /** Konstruktion. */
    public Manager(Eigenschaften eigenschaften, Statistik statistik, Fenster fenster, Ende ende, Update update) {
        this.statistik = statistik;
        this.fenster = fenster;
        this.ende = ende;

        vorgaengerspielerzahl = 0;
        for (int i = 0; i < Spieler.SPIELER; i++) {
            if (!eigenschaften.getProperty("Spieler" + i).equals("")) {
                vorgaengerspielerzahl++;
            }
        }

        Becher becher = new Becher();
        Vorschlagen vorschlagen = new Vorschlagen();
        Setzen setzen = new Setzen();
        Verfolgung verfolgung = new Verfolgung();

        Report report = new Report(statistik);
        Bestenliste bestenliste = new Bestenliste(statistik, variantenindex(eigenschaften));

        absprache = new Absprache(eigenschaften, this);
        if (eigenschaften.pfad() != null) {
            fortsetzung = new Fortsetzung(eigenschaften, fenster);
        } else {
            fortsetzung = null;
        }
        spielplatz = new Spielplatz(becher, verfolgung, setzen);
        aufsicht = new Aufsicht(eigenschaften, vorschlagen, becher, verfolgung, setzen);
        menue = new Menue(eigenschaften, vorschlagen, setzen, report, bestenliste, fortsetzung, update);

        spielplatz.addPropertyChangeListener("Beginnen", absprache);
        menue.addPropertyChangeListener("LookAndFeel", new Oberflaeche(eigenschaften));
        menue.addPropertyChangeListener("Beginnen", absprache);
        menue.addPropertyChangeListener("Karte", spielplatz);

        statistik.addChangeListener(bestenliste);
        statistik.addChangeListener(report);
        statistik.addChangeListener(new Kommentierung(spielplatz));
        fenster.zeige(eigenschaften, menue, spielplatz); // Zeige vor der automatischen Fortsetzung das Fenster!

        // Behandle Fortsetzung zuletzt, wenn die Infrastruktur aufgebaut ist.
        if (fortsetzung != null) {
            fortsetzung.addPropertyChangeListener("Speichern", menue);
            fortsetzung.verbinde(this);
            fortsetzung.versuche();
        }

    }

    /**
     * Betreibt ein neues Turnier.
     *
     * @param neu
     *            Das neu zu betreuende Turnier.
     */
    public void betreibe(Turnier neu) {
        if (turnier != null) {
            vorgaengerspielerzahl = turnier.spieler().size();
            turnier.beende();
            turnier.removeChangeListener(aufsicht);
            turnier.removeChangeListener(spielplatz);
            turnier.removeChangeListener(menue);
            turnier.removeChangeListener(ende);
            turnier.auswertung().removeChangeListener(spielplatz);
            for (Wuerfel w : turnier.wuerfel()) {
                w.entferneChangeListener(spielplatz);
            }
            turnier.removeChangeListener(fortsetzung);
        }

        if (fenster != null) {
            fenster.setTitle("JaFuffy " + "(" + neu.auswertung() + ")");
            if (vorgaengerspielerzahl != neu.spieler().size()) {
                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        fenster.pack();
                    }
                });
            }
        }

        turnier = neu;
        turnier.setzeFort(statistik);
        turnier.addChangeListener(spielplatz);
        turnier.addChangeListener(menue);
        turnier.addChangeListener(ende);
        turnier.auswertung().addChangeListener(spielplatz);
        for (Wuerfel w : turnier.wuerfel()) {
            w.installiereChangeListener(spielplatz);
        }
        turnier.addChangeListener(fortsetzung);
        turnier.addChangeListener(aufsicht);
        turnier.starte();
    }

    /**
     * Ermittelt die Variante der Spielregeln aus den Eigenschaften.
     *
     * @param eigenschaften
     *            Die persistenten Eigenschaften.
     * @return Ein Index, passend zur Aufzhlung #Variante.
     */
    private int variantenindex(Eigenschaften eigenschaften) {
        int variante;
        try {
            variante = Variante.valueOf(eigenschaften.getProperty("Variante")).ordinal();
        } catch (IllegalArgumentException exeption) {
            variante = Variante.KLASSISCH.ordinal();
            System.out.println(exeption);
        }
        return variante;
    }
}
