/******************************************************************************
 ** $Id: Kategorie.java 1647 2019-01-12 12:34:18Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

/** Zeilenarten der Tabelle (Eintrge, Kategorien). */
public enum Kategorie {

    /** Augenzahl Eins wird gezhlt. */
    EINS("1er", "Nur Einser zhlen", Art.OBEN),
    /** Augenzahl Zwei wird gezhlt. */
    ZWEI("2er", "Nur Zweier zhlen", Art.OBEN),
    /** Augenzahl Drei wird gezhlt. */
    DREI("3er", "Nur Dreier zhlen", Art.OBEN),
    /** Augenzahl Vier wird gezhlt. */
    VIER("4er", "Nur Vierer zhlen", Art.OBEN),
    /** Augenzahl Fnf wird gezhlt. */
    FUENF("5er", "Nur Fnfer zhlen", Art.OBEN),
    /** Augenzahl Sechs wird gezhlt. */
    SECHS("6er", "Nur Sechser zhlen", Art.OBEN),
    /** Die Gesamtpunktzahl der oberen Eintrge, bei denen je eine Augenzahl zum Ergebnis beitrgt. */
    GESAMT("Gesamt", "Summe der oberen Eintrge"),
    /** Bonus, falls Gesamtpunktzahl der oberen Eintrge 63 berschreitet. */
    BONUS("Bonus ab 63", "Ab 63 Punkten oben zustzlich 35 Bonuspunkte"),
    /** Gesamtpunktzahl der oberen Eintrge einschlielich Bonus. */
    OSUMME("Obere Summe", "Summe der oberen Eintrge plus Bonuspunkte"),
    /** Dreierpasch. */
    DREIERPASCH("Dreierpasch", "Alle Augen zhlen", Art.UNTEN),
    /** Viererpasch. */
    VIERERPASCH("Viererpasch", "Alle Augen zhlen", Art.UNTEN),
    /** Full-House. */
    FULLHOUSE("Full-House", "25 Punkte", Art.UNTEN),
    /** Kleine Strae. */
    KLEINESTRASSE("Kleine Strae", "30 Punkte", Art.UNTEN),
    /** Groe Strae. */
    GROSSESTRASSE("Groe Strae", "40 Punkte", Art.UNTEN),
    /** JaFuffy (fnf mal gleiche Augenzahl). */
    JAFUFFY("JaFuffy", "50 Punkte", Art.UNTEN),
    /** Chance (Summe aller Augen). */
    CHANCE("Chance", "Alle Augen zhlen", Art.UNTEN),
    /** Summe der unteren Eintrge. */
    USUMME("Untere Summe", "Summe der unteren Eintrge"),
    /** Endsumme. */
    ESUMME("Endsumme", "Obere und untere Summe zusammen");

    /** Art des Tabelleneintrages. */
    private enum Art {
        /** Der Eintrag ist nicht setzbar und enthlt abgeleitete Ergebnisse. */
        DERIVAT,
        /** Der Eintrag ist setzbar und befindet sich im oberen Tabellenteil. */
        OBEN,
        /** Der Eintrag ist setzbar und befindet sich im unteren Tabellenteil. */
        UNTEN;
    }

    /** Alle Eintrge, fr die ein Wurf gesetzt werden kann. */
    public static final Kategorie[] ALLE = {
        EINS,
        ZWEI,
        DREI,
        VIER,
        FUENF,
        SECHS,
        DREIERPASCH,
        VIERERPASCH,
        FULLHOUSE,
        KLEINESTRASSE,
        GROSSESTRASSE,
        JAFUFFY,
        CHANCE };
    /** Die Anzahl aller Tabelleneintrge. */
    public static final int ANZAHL = values().length;
    /** Alle oberen Eintrge. */
    public static final Kategorie[] OBEN = { EINS, ZWEI, DREI, VIER, FUENF, SECHS };
    /** Alle unteren Eintrge. */
    public static final Kategorie[] UNTEN = {
        DREIERPASCH,
        VIERERPASCH,
        FULLHOUSE,
        KLEINESTRASSE,
        GROSSESTRASSE,
        JAFUFFY,
        CHANCE };

    /**
     * Wandelt Nummerierung in Wert um.
     *
     * @param nummer
     *            Nummer des Aufzhlungswertes.
     * @return Aufzhlungswert.
     */
    public static Kategorie kategorie(int nummer) {
        return values()[nummer];
    }

    /**
     * Wandelt Augenzahl in Aufzhlungswert um.
     *
     * @param augen
     *            Augenzahl.
     * @return Aufzhlungswert.
     */
    public static Kategorie oben(int augen) {
        return kategorie(augen - 1);
    }

    /** Art des Tabelleneintrages. */
    private final Art art;
    /** Tabelleneintragsbeschreibung. */
    private final String beschreibung;
    /** Tabelleneintragsbezeichung. */
    private final String bezeichnung;

    /**
     * Erstellt Aufzhlungswert.
     *
     * @param bezeichnung
     *            Tabelleneintragsbezeichung.
     * @param beschreibung
     *            Tabelleneintragsbeschreibung.
     */
    private Kategorie(String bezeichnung, String beschreibung) {
        this(bezeichnung, beschreibung, Art.DERIVAT);
    }

    /**
     * Erstellt Aufzhlungswert.
     *
     * @param bezeichnung
     *            Tabelleneintragsbezeichung.
     * @param beschreibung
     *            Tabelleneintragsbeschreibung.
     * @param art
     *            Art des Aufzhlungswertes.
     */
    private Kategorie(String bezeichnung, String beschreibung, Art art) {
        this.bezeichnung = bezeichnung;
        this.beschreibung = beschreibung;
        this.art = art;
    }

    /** Gibt an, ob Kategorie erwrfelt und gesetzt werden kann. */
    public boolean alle() {
        return oben() || unten();
    }

    /** Liefert fr die Kategorie die Beschreibung als Hilfetext. */
    public String beschreibung() {
        return beschreibung;
    }

    /** Liefert die Bezeichnung der Kategorie zur Darstellung in der Tabelle. */
    public String bezeichnung() {
        return bezeichnung;
    }

    /** Gibt an, ob die Kategorie als erwrfelter Eintrag im oberen Tabellenteil liegt. */
    public boolean oben() {
        return art == Art.OBEN;
    }

    /** Gibt an, ob die Kategorie als erwrfelter Eintrag im unteren Tabellenteil liegt. */
    public boolean unten() {
        return art == Art.UNTEN;
    }

}
