/******************************************************************************
 ** $Id: Auswertung.java 1646 2019-01-12 05:33:11Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.auswertung;

import java.io.Serializable;

import jafuffy.Eigenschaften;
import jafuffy.logik.Aenderungen;
import jafuffy.logik.CEAblauf;
import jafuffy.logik.Spieler;
import jafuffy.logik.Kategorie;
import jafuffy.logik.Wuerfel;
import jafuffy.logik.analyse.Analyse;

/**
 * Auswertung des Wurfs. Textnachrichten. Hinweise zur erwrfelten Punktzahl. Setztipps. Bei der Serialisation brauchen
 * nicht alle Daten gespeichert zu werden, diese werden rekonstruiert durch Funktionsaufruf von "gewuerfelt".
 */
public abstract class Auswertung extends Aenderungen<CEAblauf> implements Serializable {

    /** Ermglicht Serialisierung. */
    private static final long serialVersionUID = -8998187500583793899L;

    /** Enthlt Auswertungs- und Prognosedaten. */
    protected final Analyse analyse;
    /** Im Turnier benutzte Wrfel. */
    protected final Wuerfel[] wuerfelsatz;

    /** Tipps (entspricht Prognose) berechnen? */
    protected boolean sindTippsAngefordert;
    /** Aktiver Spieler. */
    protected Spieler aktiver;

    /**
     * Konstruktor.
     *
     * @param wuerfelsatz
     *            Die im Turnier benutzten Wrfel.
     * @param analyse
     *            Das zur Analyse des Wurfs benutzte Objekt.
     */
    protected Auswertung(Wuerfel[] wuerfelsatz, Analyse analyse) {
        this.wuerfelsatz = wuerfelsatz;
        this.analyse = analyse;
    }

    /** @return Abweichung als grobe Schtzung von mittlerer Punktzahl pro Spiel */
    public abstract int abweichung();

    /** @return Gibt den gerade aktiven Spieler zurck. */
    public Spieler aktiver() {
        return aktiver;
    }

    /**
     * @param tipps
     *            Tipps/Prognosen geben?
     */
    public void aktiviereTipps(boolean tipps) {
        if (tipps && !sindTippsAngefordert) {
            analyse.prognostiziere(aktiver);
            sindTippsAngefordert = true;
        }
        sindTippsAngefordert = tipps;
    }

    /** @return Gibt das von der Auswertung benutzte Analyseobjekt zurck. */
    public Analyse analyse() {
        return analyse;
    }

    /** Erstelle den besten Auswahlvorschlag nach Wrfeln. */
    public void erstelleVorschlag() {
        aktiviereTipps(true);
        erstelleVorschlag(analyse.bester());
    }

    /**
     * Erstellt einen Auswahlvorschlag der Wrfel nach Wrfeln fr einen bestimmten Eintrag.
     *
     * @param kategorie
     *            Eintrag, fr den der Vorschlag erstellt werden soll.
     */
    public void erstelleVorschlag(Kategorie kategorie) {
        for (Wuerfel wuerfel : wuerfelsatz) {
            wuerfel.setzeVorschlag(analyse.erneut(kategorie, wuerfel));
        }
        fireStateChanged(CEAblauf.VORGESCHLAGEN);
    }

    /** @return Mittlere Punktzahl als grobe Schtzung pro Spiel */
    public abstract int mittelwert();

    /**
     * @param spieler
     *            Spieler, der gewrfelt hat.
     */
    public void reagiereAufWurf(Spieler spieler) {
        this.aktiver = spieler;
        sindTippsAngefordert = false;
        analyse.errechne(spieler);
    }

    /** @return Gibt an, ob Tipps angefordert wurden. */
    public boolean sindTippsAngefordert() {
        return sindTippsAngefordert;
    }

    /**
     * @param kategorie
     *            Ein noch nicht gesetzter Eintrag.
     * @return Tipp als Hilfstext ("Spicker"), der angibt, wie viele Punkte fr aktiven Spieler erreichbar sind.
     */
    public String spicker(Kategorie kategorie) {
        String text;
        if (Eigenschaften.GLOBAL.aktiv("Spicker") && aktiver.gewuerfelt() && aktiver.setzbar(kategorie)
                && kategorie.alle()) {
            int punkte = analyse.wert(kategorie);
            text = Integer.valueOf(punkte).toString() + (punkte != 1 ? " Punkte" : " Punkt");
            if (sindTippsAngefordert()) {
                text = "<html>" + "<p>" + text + "</p>" + "<p><small><i>Siehe auch Kontextmen!</i></small></p>"
                        + "<p><small><i>Beachte Men \"Extras\".</i></small></p>" + "</html>";
            }
        } else {
            text = null;
        }
        return text;
    }

    /**
     * @param kategorie
     *            Zeilennummer des Eintrags
     * @return Texteintrag in Tabelle (fr aktiven Spieler)
     */
    public abstract String text(Kategorie kategorie);

    @Override
    public abstract String toString();

}
