/******************************************************************************
 ** $Id: Punkte.java 1646 2019-01-12 05:33:11Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import java.util.ArrayList;

import javax.swing.event.TableModelEvent;
import javax.swing.table.AbstractTableModel;

import jafuffy.logik.auswertung.Auswertung;

/** Modell fr die Tabelle auf dem Zettel, gibt den Turnierstand wieder. */
@SuppressWarnings("serial")
public class Punkte extends AbstractTableModel {

    /** Alle Teilnehmer des Turniers. */
    private final ArrayList<Spieler> teilnehmer;
    /** Die verwendete Auswertung. */
    private final Auswertung auswertung;
    /** Bestimmt, ob Eintrge zum Setzen auswhlbar sind. */
    private boolean auswaehlbar = true;

    /**
     * Erstellt das Tabellenmodell.
     *
     * @param teilnehmer
     *            Alle Teilnehmer des Turniers.
     * @param auswertung
     *            Verwendete Auswertung.
     */
    Punkte(ArrayList<Spieler> teilnehmer, Auswertung auswertung) {
        this.teilnehmer = teilnehmer;
        this.auswertung = auswertung;
    }

    /**
     * Meldet zurck, ob Spieler gerade aktiv ("dran") ist und fr die Punktzahlbestimmung einer Spalte magebend ist.
     *
     * @param index
     *            Index im Teilnehmerfeld (entspricht Spalte)
     * @return Spielerstatus, sprich ob Spieler gerade aktiv ist.
     */
    public boolean aktiv(int index) {
        Spieler spieler = teilnehmer.get(index);
        return (spieler.rest() > 0 || spieler.gewuerfelt());
    }

    /**
     * Gibt an, ob ein Spieler einen Spielzug angefangen hat.
     *
     * @param index
     *            Index im Teilnehmerfeld (entspricht Spalte)
     * @return Ist der Spielzug noch nicht abgeschlossen und daher der Ausgang noch offen?
     */
    public boolean angefangen(int index) {
        return aktiv(index) && teilnehmer.get(index).rest() < Turnier.RUNDEN;
    }

    /**
     * Zellen (nicht) auswhlbar machen.
     *
     * @param auswaehlbar
     *            Schalter zur Auswhlbarkeit.
     */
    public void auswaehlbar(boolean auswaehlbar) {
        this.auswaehlbar = auswaehlbar;
    }

    /**
     * Rckmeldung der verwendeten Auswertung.
     *
     * @return Auswertungsobjekt
     */
    public Auswertung auswertung() {
        return auswertung;
    }

    /**
     * Benachrichtige registrierte Interessenten ber Vernderungen der Tabelle in der gegebenen Spalte.
     * 
     * @param index
     *            Index im Teilnehmerfeld (entspricht Spalte)
     */
    public void benachrichtige(int index) {
        fireTableChanged(new TableModelEvent(this, 0, getRowCount() - 1, index));
    }

    @Override
    public int getColumnCount() {
        return teilnehmer.size();
    }

    @Override
    public String getColumnName(int spalte) {
        return teilnehmer.get(spalte).toString();
    }

    @Override
    public int getRowCount() {
        return Kategorie.ANZAHL;
    }

    @Override
    public Object getValueAt(int zeile, int spalte) {
        return teilnehmer.get(spalte).punkte(Kategorie.kategorie(zeile));
    }

    @Override
    public boolean isCellEditable(int zeile, int spalte) {
        Spieler spieler = teilnehmer.get(spalte);
        return spieler.gewuerfelt() && spieler.setzbar(Kategorie.kategorie(zeile)) && auswaehlbar;
    }

    /**
     * Gibt an, ob der Ausgang fr einen Spielzug noch offen ist und der aktive Spieler weiter wrfeln darf.
     *
     * @param index
     *            Index im Teilnehmerfeld (entspricht Spalte)
     * @return Ist der Spielzug noch nicht abgeschlossen und daher der Ausgang noch offen?
     */
    public boolean offen(int index) {
        return teilnehmer.get(index).rest() > 0;
    }

    /**
     * Gibt an, ob ein Eintrag noch setzbar ist.
     *
     * @param zeile
     *            Tabellenzeile
     * @param spalte
     *            Tabellenspalte
     * @return Setzbarkeit
     */
    public boolean setzbar(int zeile, int spalte) {
        return teilnehmer.get(spalte).setzbar(Kategorie.kategorie(zeile));
    }

}
