/******************************************************************************
 ** $Id: Erweitert.java 1646 2019-01-12 05:33:11Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.analyse;

import jafuffy.logik.Spieler;
import jafuffy.logik.Kategorie;
import jafuffy.logik.Wuerfel;

/** Erstellt Analysen und berechnet Prognosen fr die erweiterte Variante der Spielregeln. */
public class Erweitert extends Klassisch {

    /** Kleine Strae mit der Im-Hieb-Regel. */
    class KleineStrasse extends Klassisch.KleineStrasse {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -244259498879083805L;

        @Override
        protected void bewerte(Spieler spieler) {
            if (istJokerVorliegend(spieler)) {
                wert = PUNKTE;
            } else {
                super.bewerte(spieler);
            }
        }
    }

    /** Groe Strae mit der Im-Hieb-Regel. */
    class GrosseStrasse extends Klassisch.GrosseStrasse {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -4794832240142053236L;

        @Override
        protected void bewerte(Spieler spieler) {
            if (istJokerVorliegend(spieler)) {
                wert = PUNKTE;
            } else {
                super.bewerte(spieler);
            }
        }
    }

    /** JaFuffy mit den erweiterten Regeln. */
    class JaFuffy extends Klassisch.JaFuffy {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -1936696859004502372L;

        @Override
        protected void bewerte(Spieler spieler) {
            super.bewerte(spieler);
            if (wert > 0 && spieler.punkte(Kategorie.JAFUFFY) > 0) {
                wert = spieler.punkte(Kategorie.JAFUFFY) + JOKER;
            }
        }
    }

    /** Alle Eintrge fr die erweiterte Variante der Spielregeln als Tabelle zusammengefasst. */
    class Eintragungen extends Analyse.Eintragungen {
        /** Erlaubt Serialisierung. */
        private static final long serialVersionUID = -3746750646308026451L;

        /** Konstruktion der Tabelle. */
        Eintragungen() {
            put(Kategorie.DREIERPASCH, new Dreierpasch());
            put(Kategorie.VIERERPASCH, new Viererpasch());
            put(Kategorie.FULLHOUSE, new FullHouse());
            put(Kategorie.KLEINESTRASSE, new KleineStrasse());
            put(Kategorie.GROSSESTRASSE, new GrosseStrasse());
            put(Kategorie.JAFUFFY, new JaFuffy());
        }
    }

    /** Joker-Bonus. */
    private static final int JOKER = 100;

    /** Zur Serialisierung. */
    private static final long serialVersionUID = -3064026868041038008L;

    /**
     * Konstruktor.
     *
     * @param wuerfelsatz
     *            Wrfel, mit denen das Turnier durchgefhrt wird.
     */
    public Erweitert(Wuerfel[] wuerfelsatz) {
        super(wuerfelsatz);
    }

    /**
     * @param spieler
     *            Der Spieler, dessen Wurf berprft wird.
     *
     * @return Ermittelt, ob ein Joker erwrfelt wurde.
     */
    public boolean istJokerVorliegend(Spieler spieler) {
        return paschkarte.get(Paschaufzaehlung.FUENFERPASCH) && spieler.punkte(Kategorie.JAFUFFY) > 0
                && !spieler.setzbar(Kategorie.oben(wuerfelsatz[0].augen()));
    }

    /**
     * @param spieler
     *            Der Spieler, dessen Wurf berprft wird.
     *
     * @return Gibt an, ob ein Zwang zum Setzen vorliegt (JaFuffy erwrfelt, JaFuffy-Feld belegt, oben frei).
     */
    public boolean istSetzzwangGegeben(Spieler spieler) {
        return paschkarte.get(Paschaufzaehlung.FUENFERPASCH) && spieler.punkte(Kategorie.JAFUFFY) > 0
                && spieler.setzbar(Kategorie.oben(wuerfelsatz[0].augen()));
    }

    @Override
    protected Analyse.Eintragungen eintragungen() {
        return new Eintragungen();
    }

}
