/******************************************************************************
 ** $Id: Status.java 1017 2016-05-28 20:35:09Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.BorderLayout;
import java.awt.Font;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jafuffy.bedienung.Wuerfelfeld.Kontextmenue;
import jafuffy.logik.CEAblauf;
import jafuffy.logik.CEJaFuffy;
import jafuffy.logik.Spieler;
import jafuffy.logik.Turnier;

/** Statusfeld des Wrfelfeldes, welches Icon, Nachricht und Restanzeige umfasst. */
@SuppressWarnings("serial")
class Status extends JPanel implements ChangeListener {

    /** Schriftgre fr Nachrichtenzeile. */
    private static final int NACHRICHTENSCHRIFTGROESSE = 12;
    /** Zwischenabstand zwischen einzelnen Elementen. */
    private static final int VERTIKALABSTAND = 4;

    /** ber dieses Feld lassen sich die Aktionen eines Spielers verfolgen. */
    private final Verfolgung verfolgung;
    /** Nachrichten an den Spieler. */
    private final JLabel nachricht = new JLabel();
    /** Restanzeige der verbleibenden Wrfelmglichkeiten. */
    private final JLabel restanzeige = new JLabel();

    /**
     * Erstellt Statusfeld.
     *
     * @param kontextmenue
     *            Das Kontextmen zum Statusfeld.
     * @param verfolgung
     *            Das Feld, welches die Verfolgung der Bot-Zge ermglicht.
     */
    Status(Kontextmenue kontextmenue, Verfolgung verfolgung) {
        super(new BorderLayout());
        this.verfolgung = verfolgung;
        verfolgung.addMouseListener(kontextmenue);
        dekoriere(kontextmenue);
        add(baue(), BorderLayout.CENTER);
    }

    @Override
    public void stateChanged(ChangeEvent ce) {
        if (CEJaFuffy.adressiert(ce, CEAblauf.class)) {
            CEJaFuffy<CEAblauf> cej = CEJaFuffy.ereignisbehaelter(ce);
            switch (cej.ereignis()) {
            case START:
            case SPIEL:
            case GESETZT:
            case GEWUERFELT:
            case RUECKGAENGIG:
                aktualisiere((Turnier) ce.getSource());
                break;
            case RESULTAT:
                beendeSpiel();
                break;
            case ABBRUCH:
            case ENDE:
                beendeTurnier();
                break;
            default:
                break;
            }
            verfolgung.stateChanged(ce);
        }
    }

    /**
     * Infos bezglich aktuellem Spieler; (De)Aktivierung diverser Elemente.
     *
     * @param turnier
     *            Das Turnier, fr welches der Status aktualisiert wird.
     */
    private void aktualisiere(Turnier turnier) {
        Spieler aktiver = turnier.aktiver();
        int rest = aktiver.rest();
        boolean fertig = rest == 0;
        nachricht.setText(aktiver.toString() + " ist an der Reihe!");
        nachricht.setFont(new Font("Dialog", Font.BOLD, NACHRICHTENSCHRIFTGROESSE));
        if (fertig) {
            restanzeige.setText("Bitte in der Tabelle setzen!");
        } else {
            restanzeige.setText(rest + " mal darf noch gewrfelt werden.");
        }
    }

    /**
     * Aufbau der Informationen zur Spielphase.
     *
     * @return Statusfeld.
     */
    private JPanel baue() {
        JPanel elemente = new JPanel();
        elemente.setLayout(new BoxLayout(elemente, BoxLayout.Y_AXIS));
        elemente.add(Box.createVerticalGlue());
        elemente.add(Box.createVerticalStrut(VERTIKALABSTAND));
        elemente.add(verfolgung);
        elemente.add(Box.createVerticalStrut(VERTIKALABSTAND));
        elemente.add(Box.createVerticalGlue());
        elemente.add(nachricht);
        elemente.add(Box.createVerticalStrut(VERTIKALABSTAND));
        elemente.add(restanzeige);
        elemente.add(Box.createVerticalGlue());
        return elemente;
    }

    /** Beendet das Spiel und zeigt dies ber den Status an. */
    private void beendeSpiel() {
        nachricht.setText(" ");
        restanzeige.setText(" ");
    }

    /** Beendet das Turnier und zeigt dies ber den Status an. */
    private void beendeTurnier() {
        nachricht.setText("Das Turnier ist beendet!");
        nachricht.setFont(new Font("Dialog", Font.ITALIC | Font.BOLD, NACHRICHTENSCHRIFTGROESSE));
        restanzeige.setText(" ");
    }

    /**
     * Dekoriert die Statusbestandteile Icon, Nachricht und Restanzeige und bindet Kontextmen an Icon.
     *
     * @param kontextmenue
     *            Das Kontextmen zum Icon.
     */
    private void dekoriere(Kontextmenue kontextmenue) {
        nachricht.setAlignmentX(CENTER_ALIGNMENT);
        restanzeige.setAlignmentX(CENTER_ALIGNMENT);
    }

}
