/******************************************************************************
 ** $Id: Manager.java 1772 2019-06-30 20:16:16Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import javax.swing.SwingUtilities;

import jafuffy.Eigenschaften;
import jafuffy.bedienung.Absprache;
import jafuffy.bedienung.Becher;
import jafuffy.bedienung.Bestenliste;
import jafuffy.bedienung.Ende;
import jafuffy.bedienung.Fenster;
import jafuffy.bedienung.Fortsetzung;
import jafuffy.bedienung.Kommentierung;
import jafuffy.bedienung.Menue;
import jafuffy.bedienung.Oberflaeche;
import jafuffy.bedienung.Report;
import jafuffy.bedienung.Setzen;
import jafuffy.bedienung.Spielplatz;
import jafuffy.bedienung.Verfolgung;
import jafuffy.bedienung.Vorschlagen;

/** Stellt die Infrastruktur zum Betrieb eines Turniers. */
public class Manager {

    /** Die Absprache zu einem neuen Turnier. */
    private final Absprache absprache;
    /** Sicherheitsfrage zur Besttigung der Programmbeendigung. */
    private final Ende ende;
    /** Fenster, in dem JaFuffy abluft. */
    private final Fenster fenster;
    /** Kmmert sich um die Fortsetzung von gespeicherten Turnieren. */
    private final Fortsetzung fortsetzung;
    /** Men im Fenster. */
    private final Menue menue;
    /** Spielplatz mit Punktezettel und Wrfelfeld. */
    private final Spielplatz spielplatz;
    /** bernimmt die Aufsicht ber die einzelnen Schritte der Bot-Aktionen. */
    private final Aufsicht aufsicht;
    /** Die Statistik, welche fr alle Turnier gefhrt wird. */
    private final Statistik statistik;

    /** Speichert das Turnier, welches zurzeit durch den Manager betrieben wird. */
    private Turnier turnier;
    /**
     * Anzahl der Spieler im letzten Turnier, welche bei Start von JaFuffy der automatischen Sicherung entnommen wird.
     */
    private int vorgaengerspielerzahl;

    /** Konstruktion. */
    public Manager(Eigenschaften eigenschaften, Statistik statistik, Fenster fenster, Ende ende, Update update) {
        this.statistik = statistik;
        this.fenster = fenster;
        this.ende = ende;

        vorgaengerspielerzahl = 0;
        for (int i = 0; i < Spieler.SPIELER; i++) {
            if (!eigenschaften.getProperty("Spieler" + i).equals("")) {
                vorgaengerspielerzahl++;
            }
        }

        absprache = new Absprache(eigenschaften, this);
        if (eigenschaften.pfad() != null) {
            fortsetzung = new Fortsetzung(eigenschaften, fenster);
        } else {
            fortsetzung = null;
        }

        final Becher becher = new Becher();
        final Verfolgung verfolgung = new Verfolgung();
        final Setzen setzen = new Setzen();
        final Vorschlagen vorschlagen = new Vorschlagen();
        final Report report = new Report(statistik);
        final Bestenliste bestenliste = new Bestenliste(statistik, variantenindex(eigenschaften));

        spielplatz = new Spielplatz(eigenschaften, becher, verfolgung, setzen);
        aufsicht = new Aufsicht(eigenschaften, vorschlagen, becher, verfolgung, setzen);
        menue = new Menue(eigenschaften, vorschlagen, setzen, report, bestenliste, fortsetzung, update);

        spielplatz.addPropertyChangeListener("Beginnen", absprache);
        menue.addPropertyChangeListener("LookAndFeel", new Oberflaeche(eigenschaften));
        menue.addPropertyChangeListener("Beginnen", absprache);
        menue.addPropertyChangeListener("Karte", spielplatz);

        statistik.addChangeListener(bestenliste);
        statistik.addChangeListener(report);
        statistik.addChangeListener(new Kommentierung(spielplatz));
        fenster.zeige(eigenschaften, menue, spielplatz); // Zeige vor der automatischen Fortsetzung das Fenster!

        // Behandle Fortsetzung zuletzt, wenn die Infrastruktur aufgebaut ist.
        if (fortsetzung != null) {
            fortsetzung.addPropertyChangeListener("Speichern", menue);
            fortsetzung.verbinde(this);
            fortsetzung.versuche();
        }

    }

    /**
     * Betreibt ein neues Turnier.
     *
     * @param neu
     *            Das neu zu betreuende Turnier.
     */
    public void betreibe(Turnier neu) {
        if (turnier != null) {
            vorgaengerspielerzahl = turnier.teilnehmer().size();
            turnier.beende();
            turnier.removeChangeListener(aufsicht);
            turnier.removeChangeListener(spielplatz);
            turnier.removeChangeListener(menue);
            turnier.removeChangeListener(ende);
            turnier.auswertung().removeChangeListener(spielplatz);
            for (Wuerfel wuerfel : turnier.wuerfelsatz()) {
                wuerfel.entferneChangeListener(spielplatz);
            }
            turnier.removeChangeListener(fortsetzung);
        }

        if (fenster != null) {
            fenster.setTitle("JaFuffy " + "(" + neu.auswertung() + ")");
            if (vorgaengerspielerzahl != neu.teilnehmer().size()) {
                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        fenster.pack();
                    }
                });
            }
        }

        turnier = neu;
        turnier.setzeFort(statistik);
        turnier.addChangeListener(spielplatz);
        turnier.addChangeListener(menue);
        turnier.addChangeListener(ende);
        turnier.auswertung().addChangeListener(spielplatz);
        for (Wuerfel wuerfel : turnier.wuerfelsatz()) {
            wuerfel.installiereChangeListener(spielplatz);
        }
        turnier.addChangeListener(fortsetzung);
        turnier.addChangeListener(aufsicht);
        turnier.starte();
    }

    /**
     * Ermittelt die Variante der Spielregeln aus den Eigenschaften.
     *
     * @param eigenschaften
     *            Die persistenten Eigenschaften.
     * @return Ein Index, passend zur Aufzhlung #Variante.
     */
    private int variantenindex(Eigenschaften eigenschaften) {
        int variante;
        try {
            variante = Variante.valueOf(eigenschaften.getProperty("Variante")).ordinal();
        } catch (IllegalArgumentException exeption) {
            variante = Variante.KLASSISCH.ordinal();
            System.out.println(exeption);
        }
        return variante;
    }
}
