/******************************************************************************
 ** $Id: Menue.java 2584 2021-01-17 10:40:41Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/

package jafuffy.bedienung;

import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.UIManager;
import javax.swing.UIManager.LookAndFeelInfo;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jafuffy.Eigenschaften;
import jafuffy.Ressource;
import jafuffy.logik.Turnier;
import jafuffy.logik.Update;
import jafuffy.logik.ereignis.Ablauf;
import jafuffy.logik.ereignis.Umschlag;

/** Erzeugung der Menzeile und Einrichtung der Ereignislauscher der Meneintrge. */
@SuppressWarnings("serial")
public class Menue extends JMenuBar implements PropertyChangeListener, ChangeListener {

    /** Wechseln von Eigenschaften. */
    private class Wechsel implements ActionListener {
        private final String schluessel;

        Wechsel(String schluessel) {
            this.schluessel = schluessel;
        }

        @Override
        public void actionPerformed(ActionEvent ereignis) {
            eigenschaften.wechsle(schluessel);
        }

        boolean aktiv() {
            return eigenschaften.aktiv(schluessel);
        }
    }

    /** Fehlermeldung fr Applet. */
    private static final String SANDBOX = "<html>"
            + "<p><strong>Funktion auer Aufnahme:</strong> <i>JaFuffy</i> luft mit eingeschrnkten Rechten.</p>"
            + "<p>Dies ist u.a. der Fall bei der Ausfhrung von <i>JaFuffy</i> per Java Web Start.</p>" + "<br>"
            + "<p>Zur Bereitstellung der Funktion bitte <i>JaFuffy</i> ber Homepage installieren.</p>" + "</html>";

    /** Dialog zur Anzeige des Turnierverlaufs. */
    private final Report report;
    /** Dialoge zur Anzeige der Bestenliste. */
    private final Bestenliste bestenliste;
    /** Dialog zur Fortsetzung von Turnieren. */
    private final Fortsetzung fortsetzung;
    /** berprfung auf Update von JaFuffy. */
    private final Update update;
    /** Dialog zur Einstellungen der Spielstrken der einzelnen Bots. */
    private final Botstaerke botstaerke;
    /** Rechtliche Informationen, Versionsangabe. */
    private final Info info = new Info();
    /** Hinweise zur Lizenzierung. */
    private final Lizenzierung lizenzierung = new Lizenzierung();

    private final JMenuItem speichern = new JMenuItem("Speichern");
    private final JMenuItem speichernUnter = new JMenuItem("Speichern unter...");
    private final JMenuItem berichten = new JMenuItem("Report");
    private final JMenuItem rueckgaengig = new JMenuItem("Rckgngig");
    private final JMenuItem beenden = new JMenuItem("Beenden");
    /** Die Eigenschaften, mit denen JaFuffy betrieben wird. */
    private final Eigenschaften eigenschaften;
    /** Meneintrag zur Erstellung eines Vorschlags fr die Auswahl eines Eintrages und dazu passender Wrfel. */
    private final Vorschlagen vorschlagen;
    /** Meneintrag zum Setzen der Wrfelauswahl. */
    private final Setzen setzen;

    /** Gibt an, ob im Spielbetrieb oder noch Intro. */
    private boolean betrieb = false;
    /** Gibt an, ob im Netzwerkbetrieb. */
    private boolean netzwerk = false;

    /**
     * Erstellt Drop-Down-Men.
     *
     * @param eigenschaften
     *            Die Eigenschaften, mit denen JaFuffy betrieben wird.
     * @param vorschlagen
     *            Meneintrag zur Erstellung eines Vorschlags fr die Auswahl eines Eintrages und dazu passender Wrfel.
     * @param setzen
     *            Meneintrag zum Setzen der Wrfelauswahl.
     * @param report
     *            Dialog zur Anzeige des Turnierverlaufs.
     * @param bestenliste
     *            Dialoge zur Anzeige der Bestenliste.
     * @param fortsetzung
     *            Dialog zur Fortsetzung von Turnieren.
     * @param update
     *            berprfung auf Update von JaFuffy.
     */
    public Menue(Eigenschaften eigenschaften, Vorschlagen vorschlagen, Setzen setzen, Report report,
            Bestenliste bestenliste, Fortsetzung fortsetzung, Update update) {
        this.eigenschaften = eigenschaften;
        this.vorschlagen = vorschlagen;
        this.setzen = setzen;
        this.report = report;
        this.bestenliste = bestenliste;
        this.fortsetzung = fortsetzung;
        this.update = update;
        botstaerke = new Botstaerke(eigenschaften);

        UIManager.addPropertyChangeListener(this);

        erzeugeTurniermenue();
        erzeugeStatistikmenue();
        erzeugeExtrasmenue();
        erzeugeEinstellungenmenue();
        erzeugeHilfemenue();
    }

    @Override
    public void propertyChange(PropertyChangeEvent event) {
        String name = event.getPropertyName();
        if (name.equals("lookAndFeel")) {
            SwingUtilities.updateComponentTreeUI(this);
        } else if (name.equals("Speichern")) {
            speichern.setEnabled((boolean) event.getNewValue());
        } else if (name.equals("Netzwerk")) {
            netzwerk = (boolean) event.getNewValue();
        }
    }

    @Override
    public void stateChanged(ChangeEvent ce) {
        if (Umschlag.adressiert(ce, Ablauf.class)) {
            Umschlag<Ablauf, Turnier> umschlag = Umschlag.ereignisbehaelter(ce);
            Turnier turnier = umschlag.quelle();
            switch (umschlag.ereignis()) {
            case START:
                meldeTurnierAn(turnier);
                break;
            case RESULTAT:
                aktualisiere(false, false, false);
                break;
            case ABBRUCH:
            case ENDE:
                meldeTurnierAb(turnier);
                break;
            case RUECKGAENGIG:
            case GEWORFEN:
                aktualisiere(true, false, false);
                break;
            case FERTIG:
                aktualisiere(false, false, turnier.ruecknahmefaehig() && !netzwerk);
                break;
            default:
                break;
            }
        }
    }

    /**
     * Aktualisiert einige Meneintrge.
     *
     * @param vorschlagbar
     *            Gibt an, ob Vorschlge angefordet werden knnen.
     * @param setzbar
     *            Gibt an, ob ein Wurfergebnis setzbar ist.
     * @param ruecknahmefaehig
     *            Gibt an, ob ein Setzvorgang rckgngig gemacht werden kann.
     */
    private void aktualisiere(boolean vorschlagbar, boolean setzbar, boolean ruecknahmefaehig) {
        vorschlagen.setEnabled(vorschlagbar);
        setzen.setEnabled(setzbar);
        rueckgaengig.setEnabled(ruecknahmefaehig);
    }

    /**
     * Erstellt das Untermen, welches das Botverhalten einstellt.
     *
     * @return Meneintrag.
     */
    private JMenu botkontrolle() {
        JMenu untermenue = new JMenu("Botverhalten");
        untermenue.setToolTipText(
                "<html>Ein Bot ist ein Gegenspieler, dessen Rolle von <i>JaFuffy</i> bernommen wird.</html>");
        JMenuItem eintrag;
        eintrag = untermenue.add(new JMenuItem("Spielstrke der Bots..."));
        eintrag.setToolTipText("Spielstrke der einzelnen Bots einstellen oder ansehen");
        eintrag.addActionListener(ereignis -> {
            botstaerke.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
            botstaerke.setVisible(true);
        });
        untermenue.addSeparator();
        eintrag = untermenue.add(new JCheckBoxMenuItem("Zeitlich kurze Darstellung"));
        eintrag.setToolTipText("Falls ausgewhlt werden Aktionen des Bots zeitlich schneller dargestellt.");
        eintrag.setSelected(eigenschaften.aktiv("Kurzdarstellung"));
        eintrag.addActionListener(new Wechsel("Kurzdarstellung"));
        untermenue.addSeparator();
        ActionListener beobachter = ereignis -> {
            boolean ueberlegungsbestaetigung;
            boolean geisterhand;
            switch (ereignis.getActionCommand()) {
            case "Einzeln":
            default:
                ueberlegungsbestaetigung = true;
                geisterhand = false;
                break;
            case "Gesammelt":
                ueberlegungsbestaetigung = false;
                geisterhand = false;
                break;
            case "Geisterhand":
                ueberlegungsbestaetigung = false;
                geisterhand = true;
                break;
            }
            eigenschaften.wechsle("Ueberlegungsbestaetigung", ueberlegungsbestaetigung);
            eigenschaften.wechsle("Geisterhand", geisterhand);
        };
        ButtonGroup gruppe = new ButtonGroup();
        eintrag = untermenue.add(new JRadioButtonMenuItem("berlegungen einzeln besttigen"));
        eintrag.setActionCommand("Einzeln");
        gruppe.add(eintrag);
        eintrag.setToolTipText(
                "<html>" + "berlegungen zur Wrfelauswahl mssen zustzlich einzeln besttigt werden." + "</html>");
        eintrag.setSelected(eigenschaften.aktiv("Ueberlegungsbestaetigung") && !eigenschaften.aktiv("Geisterhand"));
        eintrag.addActionListener(beobachter);
        eintrag = untermenue.add(new JRadioButtonMenuItem("Aktionen gesammelt besttigen"));
        eintrag.setActionCommand("Gesammelt");
        gruppe.add(eintrag);
        eintrag.setToolTipText("<html>" + "Aktionen werden zusammengefasst besttigt." + "</html>");
        eintrag.setSelected(!eigenschaften.aktiv("Ueberlegungsbestaetigung") && !eigenschaften.aktiv("Geisterhand"));
        eintrag.addActionListener(beobachter);
        eintrag = untermenue.add(new JRadioButtonMenuItem("Keinerlei Besttigung erforderlich."));
        eintrag.setActionCommand("Geisterhand");
        gruppe.add(eintrag);
        eintrag.setToolTipText("<html>" + "<p>Bots spielen vollstndig autonom, "
                + "ohne dass menschliche Mitspieler eingreifen mssen oder knnen.</p>"
                + "<p><em>Achtung:</em> Der vorhergehende Spieler kann sein Setzen nicht rckgngig machen.</p>"
                + "<html>");
        eintrag.setSelected(eigenschaften.aktiv("Geisterhand"));
        eintrag.addActionListener(beobachter);
        return untermenue;
    }

    /**
     * Erstellt das Untermen zur Einstellung der Grafik- und Toneffekte.
     *
     * @return Meneintrag
     */
    private JMenu effekte() {
        JMenu untermenue = new JMenu("Effekte");
        untermenue.setToolTipText("Grafik- und Toneffekte");
        JMenuItem eintrag;
        eintrag = untermenue.add(new JCheckBoxMenuItem("Eingangsgrafik"));
        eintrag.setToolTipText("Eingangsgrafik ein/ausschalten");
        eintrag.setSelected(eigenschaften.aktiv("Intro"));
        eintrag.addActionListener(new Wechsel("Intro") {
            @Override
            public void actionPerformed(ActionEvent ereignis) {
                if (!betrieb) {
                    super.actionPerformed(ereignis);
                    if (aktiv()) {
                        firePropertyChange("Karte", Spielplatz.TEXT, Spielplatz.INTRO);
                    } else {
                        firePropertyChange("Karte", Spielplatz.INTRO, Spielplatz.TEXT);
                    }
                }
            }
        });
        eintrag = untermenue.add(new JCheckBoxMenuItem("Gerusche"));
        eintrag.setEnabled(Ressource.soundsystem());
        eintrag.setToolTipText("Gerusche ein/ausschalten");
        eintrag.setSelected(Ressource.soundsystem() && eigenschaften.aktiv("Geraeusche"));
        eintrag.addActionListener(new Wechsel("Geraeusche"));
        return untermenue;
    }

    /**
     * Erstellt den Meneintrag zum Einfachklick.
     *
     * @return Meneintrag
     */
    private JMenuItem einfachklick() {
        JMenuItem eintrag = new JCheckBoxMenuItem("Setzen durch Einfachklick");
        eintrag.setToolTipText("<html>" + "Wahl zwischen einmaligem und zweimaligem Klicken" + "<br>"
                + "zum Setzen des Wurfs in der Tabelle" + "</html>");
        eintrag.setSelected(eigenschaften.aktiv("Einfachklick"));
        eintrag.addActionListener(new Wechsel("Einfachklick"));
        return eintrag;
    }

    /** Erzeugt das Einstellungsmen. */
    private void erzeugeEinstellungenmenue() {
        JMenu menue = new JMenu("Einstellungen");
        menue.setMnemonic('E');
        menue.add(einfachklick());
        menue.add(botkontrolle());
        menue.addSeparator();
        menue.add(zwischenstand());
        menue.add(meldungen());
        menue.add(effekte());
        menue.add(lookandfeel());
        menue.addSeparator();
        menue.add(pfadwahl());
        menue.addSeparator();
        menue.add(versionsnachfrage());
        add(menue);
    }

    /** Erzeugung eines Mens. */
    private void erzeugeExtrasmenue() {
        JMenu menue = new JMenu("Extras");
        menue.setMnemonic('X');
        add(menue);

        JMenuItem eintrag;
        menue.add(rueckgaengig);

        rueckgaengig.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, 0));
        rueckgaengig.setActionCommand("Rueckgaengig");
        rueckgaengig.setEnabled(false);
        rueckgaengig.setToolTipText("Letztes Setzen rckgngig machen");
        menue.add(setzen);
        //
        menue.addSeparator();
        //
        eintrag = menue.add(new JMenuItem("Fenster anpassen"));
        eintrag.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, 0));
        eintrag.setToolTipText("Fenstergre optimal anpassen " + "an Tabelle, Wrfelfeld und Look & Feel");
        eintrag.addActionListener(ereignis -> SwingUtilities.invokeLater(() -> {
            Window umgebung = SwingUtilities.getWindowAncestor(Menue.this);
            umgebung.pack();
        }));
        //
        menue.addSeparator();
        //
        eintrag = menue.add(new JMenuItem("Neu zeichnen"));
        eintrag.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, 0));
        eintrag.setToolTipText("Alles neu zeichnen zur Beseitigung von Darstellungsfehlern");
        eintrag.addActionListener(ereignis -> SwingUtilities.getRootPane(Menue.this).repaint());
        eintrag = menue.add(new JMenuItem("Neustart"));
        if (eigenschaften.pfad() != null) {
            eintrag.addActionListener(event -> {
                eigenschaften.merkeNeustartAnforderung(betrieb);
                beenden.doClick();
            });
            eintrag.setToolTipText("<html>" + "<p>Neustart durchfhren mit automatischer Turnierfortsetzung.</p>"
                    + "<p>Hilfreich weil Look & Feel-Wechsel nicht immer funktionieren.</p>" + "</html>");
        } else {
            eintrag.setEnabled(false);
            eintrag.setToolTipText(SANDBOX);
        }

    }

    /** Erzeugung eines Mens. */
    private void erzeugeHilfemenue() {
        JMenu menue = new JMenu("Hilfe");
        menue.setMnemonic('H');
        add(menue);

        JMenuItem eintrag;

        menue.add(vorschlagen);
        vorschlagen.addActionListener(ereignis -> setzen.setEnabled(true));
        //
        menue.addSeparator();
        //
        eintrag = menue.add(new JMenuItem("Hilfethemen"));
        eintrag.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_H, 0));
        eintrag.addActionListener(event -> {
            try {
                Hilfe hilfe = new Hilfe();
                hilfe.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                hilfe.setVisible(true);
            } catch (Exception exception) {
                exception.printStackTrace();
                Signal.WARNUNG.play();
                JOptionPane.showMessageDialog(SwingUtilities.getRootPane(Menue.this),
                        "Hilfetexte knnen nicht angezeigt werden!\n"
                                + "Fehlerhafte Installation? Keine Internetverbindung?\n" + "Java-Exception: "
                                + exception,
                        "JaFuffy (Warnung)", JOptionPane.WARNING_MESSAGE);
            }
        });
        //
        menue.addSeparator();
        //
        // ber JaFuffy
        eintrag = menue.add(new JMenuItem("<html>ber <i>JaFuffy</i></html>"));
        eintrag.addActionListener(ereignis -> {
            info.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
            info.setVisible(true);
        });
        // Lizenzierung
        eintrag = menue.add(new JMenuItem("Lizenzierung"));
        eintrag.addActionListener(ereignis -> {
            lizenzierung.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
            lizenzierung.setVisible(true);
        });
        //
        menue.addSeparator();
        //
        // Aktualisieren
        eintrag = menue.add(new JMenuItem("Aktualisieren..."));
        eintrag.setToolTipText("<html>" + "Sucht auf der <i>JaFuffy</i>-Homepage nach neuerer Version" + "<html>");
        eintrag.addActionListener(ereignis -> Menue.this.update.ueberpruefe(SwingUtilities.getRootPane(Menue.this)));
    }

    /** Erzeugung eines Mens. */
    private void erzeugeStatistikmenue() {
        JMenu menue = new JMenu("Statistik");
        menue.setMnemonic('S');
        add(menue);

        JMenuItem eintrag;

        menue.add(berichten);
        berichten.setToolTipText("(Zischen)Stand & Turnierverlauf");
        berichten.setEnabled(false);
        berichten.addActionListener(ereignis -> {
            Menue.this.report.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
            Menue.this.report.setVisible(true);
        });
        eintrag = menue.add(new JMenuItem("Bestenliste"));
        eintrag.addActionListener(ereignis -> {
            Menue.this.bestenliste.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
            Menue.this.bestenliste.setVisible(true);
        });
    }

    /** Erzeugung eines Mens. */
    private void erzeugeTurniermenue() {
        JMenu menue = new JMenu("Turnier");
        menue.setMnemonic('T');
        add(menue);

        JMenuItem eintrag;

        eintrag = menue.add(new JMenuItem("Beginnen..."));
        eintrag.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_B, 0));
        eintrag.addActionListener(ereignis -> Menue.this.firePropertyChange("Beginnen", null, Menue.this));
        //
        menue.addSeparator();
        //
        menue.add(speichern);
        speichern.setEnabled(false);
        speichern.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, 0));
        if (fortsetzung != null) {
            speichern.addActionListener(ereignis -> {
                Menue.this.fortsetzung.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                Menue.this.fortsetzung.speichere();
            });
        } else {
            speichern.setToolTipText(SANDBOX);
        }
        menue.add(speichernUnter);
        speichernUnter.setEnabled(false);
        if (fortsetzung != null) {
            speichernUnter.addActionListener(ereignis -> {
                Menue.this.fortsetzung.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                Menue.this.fortsetzung.speichereUnter();
                Menue.this.fortsetzung.setVisible(true);
            });
        } else {
            speichernUnter.setToolTipText(SANDBOX);
        }
        eintrag = menue.add(new JMenuItem("Fortsetzen..."));
        if (fortsetzung != null) {
            eintrag.addActionListener(ereignis -> {
                Menue.this.fortsetzung.setLocationRelativeTo(SwingUtilities.getRootPane(Menue.this));
                Menue.this.fortsetzung.setzeFort();
                Menue.this.fortsetzung.setVisible(true);
            });
        } else {
            eintrag.setEnabled(false);
            eintrag.setToolTipText(SANDBOX);
        }
        //
        menue.addSeparator();
        //
        menue.add(beenden);
        beenden.addActionListener(ereignis -> {
            Window umgebung = SwingUtilities.getWindowAncestor(Menue.this);
            WindowEvent fensterereignis = new WindowEvent(umgebung, WindowEvent.WINDOW_CLOSING);
            umgebung.dispatchEvent(fensterereignis);
        });
    }

    /** @return Eine alphabetisch sortierte Liste aller Look & Feel. */
    private List<LookAndFeelInfo> lafs() {
        List<LookAndFeelInfo> lafs = Arrays.asList(UIManager.getInstalledLookAndFeels());
        Collections.sort(lafs, (laf1, laf2) -> laf1.getName().compareToIgnoreCase(laf2.getName()));
        return lafs;
    }

    /**
     * Erstellt das Untermen zum Look & Feel.
     *
     * @return Meneintrag
     */
    private JMenu lookandfeel() {
        JMenu untermenue = new JMenu("Look and Feel");
        untermenue.setToolTipText("<html>" + "<p>Look & Feel der Benutzeroberflche.</p>"
                + "<p>Je nach Auswahl kann ein Neustart erforderlich sein, siehe Extras-Men.</p>" + "</html>");
        ButtonGroup gruppe = new ButtonGroup();
        for (final LookAndFeelInfo laf : lafs()) {
            JMenuItem eintrag = new JRadioButtonMenuItem(laf.getName());
            String alt = eigenschaften.getProperty("LookAndFeel");
            eintrag.setSelected(alt.equals(laf.getClassName()));
            eintrag.addActionListener(event -> {
                String alt1 = eigenschaften.getProperty("LookAndFeel");
                String neu = laf.getClassName();
                firePropertyChange("LookAndFeel", alt1, neu);
            });
            untermenue.add(eintrag);
            gruppe.add(eintrag);
        }
        return untermenue;
    }

    /**
     * Meldet Turnier ab.
     *
     * @param turnier
     *            Abzumeldendes Turnier.
     */
    private void meldeTurnierAb(Turnier turnier) {
        betrieb = false;
        aktualisiere(false, false, false);
        speichern.setEnabled(false);
        speichernUnter.setEnabled(false);
        vorschlagen.removeActionListener(turnier);
        rueckgaengig.removeActionListener(turnier);
    }

    /**
     * Meldet Turnier an.
     *
     * @param turnier
     *            Anzumeldendes Turnier.
     */
    private void meldeTurnierAn(Turnier turnier) {
        betrieb = true;
        aktualisiere(turnier.vorschlagbar(), false, false);
        vorschlagen.addActionListener(turnier);
        rueckgaengig.addActionListener(turnier);
        speichern.setEnabled(false);
        speichernUnter.setEnabled(fortsetzung != null && !netzwerk);
        berichten.setEnabled(true);
    }

    /**
     * Erstellt das Untermen zu den Meldungsoptionen.
     *
     * @return Meneintrag
     */
    private JMenu meldungen() {
        JMenu untermenue = new JMenu("Meldungen");
        untermenue.setToolTipText("Meldungen ein/ausschalten bei...");
        JMenuItem eintrag;
        eintrag = untermenue.add(new JCheckBoxMenuItem("Hinweise"));
        eintrag.setToolTipText("Allgemeine Hinweise (\"ToolTips\" in Ksten) anzeigen?");
        eintrag.setSelected(eigenschaften.aktiv("Hinweis"));
        ToolTipManager.sharedInstance().setEnabled(eigenschaften.aktiv("Hinweis"));
        eintrag.addActionListener(new Wechsel("Hinweis") {
            @Override
            public void actionPerformed(ActionEvent ereignis) {
                super.actionPerformed(ereignis);
                ToolTipManager.sharedInstance().setEnabled(aktiv());
            }
        });
        eintrag = untermenue.add(new JCheckBoxMenuItem("Spicker"));
        eintrag.setToolTipText("Spicker fr Wrfelpunkte in Tabelle?");
        eintrag.setSelected(eigenschaften.aktiv("Spicker"));
        eintrag.addActionListener(new Wechsel("Spicker"));
        eintrag = untermenue.add(new JCheckBoxMenuItem("Erinnerung abgebrochenes Turnier"));
        if (fortsetzung != null) {
            eintrag.setToolTipText("Erinnerung an abgebrochenes Turnier zum Weiterspielen?");
            eintrag.setSelected(eigenschaften.aktiv("Erinnerung"));
            eintrag.addActionListener(new Wechsel("Erinnerung"));
        } else {
            eintrag.setEnabled(false);
            eintrag.setToolTipText(SANDBOX);
        }
        return untermenue;
    }

    /**
     * Erstellt den Meneintrag zur Pfadwahl.
     *
     * @return Meneintrag
     */
    private JMenuItem pfadwahl() {
        JMenuItem eintrag = new JMenuItem("Pfad...");
        if (eigenschaften.pfad() != null) {
            eintrag.setToolTipText("<html>" + "<p>Pfad zu Statistikdaten und Turnierstnden einstellen.</p>"
                    + "<p>Aktuelle Einstellung: " + "<strong>" + eigenschaften.pfad() + "</strong></p>" + "</html>");
            eintrag.addActionListener(ereignis -> new Pfad(eigenschaften, SwingUtilities.getRootPane(Menue.this)));
        } else {
            eintrag.setEnabled(false);
            eintrag.setToolTipText(SANDBOX);
        }
        return eintrag;
    }

    /**
     * Erstellt den Meneintrag zur Versionsnachfrage.
     *
     * @return Meneintrag
     */
    private JMenuItem versionsnachfrage() {
        JMenuItem eintrag = new JCheckBoxMenuItem("Nach neuerer Version schauen");
        if (eigenschaften.pfad() != null) {
            eintrag.setToolTipText("<html>Bei jedem Programmstart automatisch "
                    + "nach neuerer Version auf <i>JaFuffy</i>-Homepage suchen<html>");
            eintrag.setSelected(eigenschaften.aktiv("Nachschauen"));
            eintrag.addActionListener(new Wechsel("Nachschauen"));
        } else {
            eintrag.setEnabled(false);
            eintrag.setToolTipText(SANDBOX);
        }
        return eintrag;
    }

    /**
     * Erstellt den Meneingrag zur Zwischenstandsanzeige.
     *
     * @return Meneintrag
     */
    private JMenuItem zwischenstand() {
        JMenuItem eintrag;
        eintrag = new JCheckBoxMenuItem("Zwischenstandsanzeige");
        eintrag.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, 0));
        eintrag.setToolTipText("<html>" + "Die Punktezahl des derzeitigen Wurfs" + "<br>"
                + "fr jeden freien Tabellenplatz anzeigen?" + "</html>");
        eintrag.setSelected(eigenschaften.aktiv("Zwischenstand"));
        eintrag.addActionListener(new Wechsel("Zwischenstand"));
        eintrag.addActionListener(ereignis -> SwingUtilities.getRootPane(Menue.this).repaint());
        return eintrag;
    }

}
