/******************************************************************************
 ** $Id: Manager.java 2596 2021-01-24 17:18:23Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import java.beans.PropertyChangeEvent;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;

import javax.swing.SwingUtilities;

import jafuffy.Eigenschaften;
import jafuffy.bedienung.Absprache;
import jafuffy.bedienung.Becher;
import jafuffy.bedienung.Bestenliste;
import jafuffy.bedienung.Ende;
import jafuffy.bedienung.Fenster;
import jafuffy.bedienung.Fortsetzung;
import jafuffy.bedienung.Kommentierung;
import jafuffy.bedienung.Menue;
import jafuffy.bedienung.Oberflaeche;
import jafuffy.bedienung.Report;
import jafuffy.bedienung.Setzen;
import jafuffy.bedienung.Spielplatz;
import jafuffy.bedienung.Verfolgung;
import jafuffy.bedienung.Vorschlagen;
import jafuffy.netzwerk.Vermittlung;
import jafuffy.netzwerk.Vermittlung.Eroeffnung;

/** Stellt die Infrastruktur zum Aufnahme eines Turniers. */
public class Manager {

    /** Die Eigenschaften fr die laufende Instanz von JaFuffy. */
    private final Eigenschaften eigenschaften;
    /** Die Absprache zu einem neuen Turnier. */
    private final Absprache absprache;
    /** Sicherheitsfrage zur Besttigung der Programmbeendigung. */
    private final Ende ende;
    /** Fenster, in dem JaFuffy abluft. */
    private final Fenster fenster;
    /** Kmmert sich um die Fortsetzung von gespeicherten Turnieren. */
    private final Fortsetzung fortsetzung;
    /** Men im Fenster. */
    private final Menue menue;
    /** Spielplatz mit Punktezettel und Wrfelfeld. */
    private final Spielplatz spielplatz;
    /** bernimmt die Aufsicht ber die einzelnen Schritte der Bot-Aktionen. */
    private final Aufsicht aufsicht;
    /** Die Statistik, welche fr alle Turnier gefhrt wird. */
    private final Statistik statistik;

    /** Speichert das Turnier, welches zurzeit durch den Manager betrieben wird. */
    private Turnier turnier;
    /**
     * Anzahl der Spieler im letzten Turnier, welche bei Start von JaFuffy der automatischen Sicherung entnommen wird.
     */
    private int vorgaengerspielerzahl;

    /** Konstruktion. */
    public Manager(Eigenschaften eigenschaften, Statistik statistik, Fenster fenster, Ende ende, Update update) {
        this.eigenschaften = eigenschaften;
        this.statistik = statistik;
        this.fenster = fenster;
        this.ende = ende;

        vorgaengerspielerzahl = 0;
        for (int i = 0; i < Spieler.SPIELER; i++) {
            if (!eigenschaften.getProperty("Name" + i).equals("")) {
                vorgaengerspielerzahl++;
            }
        }

        absprache = new Absprache(eigenschaften, fenster, this);
        if (eigenschaften.pfad() != null) {
            fortsetzung = new Fortsetzung(eigenschaften, fenster);
        } else {
            fortsetzung = null;
        }

        final Becher becher = new Becher();
        final Verfolgung verfolgung = new Verfolgung(statistik);
        final Setzen setzen = new Setzen();
        final Vorschlagen vorschlagen = new Vorschlagen();
        final Report report = new Report(statistik);
        final Bestenliste bestenliste = new Bestenliste(statistik, eigenschaften.variante());

        spielplatz = new Spielplatz(eigenschaften, becher, verfolgung, setzen);
        aufsicht = new Aufsicht(eigenschaften, fenster, vorschlagen, becher, verfolgung, setzen);
        menue = new Menue(eigenschaften, vorschlagen, setzen, report, bestenliste, fortsetzung, update);

        spielplatz.addPropertyChangeListener("Beginnen", absprache);
        menue.addPropertyChangeListener("LookAndFeel", new Oberflaeche(eigenschaften));
        menue.addPropertyChangeListener("Beginnen", absprache);
        menue.addPropertyChangeListener("Karte", spielplatz);

        statistik.addChangeListener(bestenliste);
        statistik.addChangeListener(report);
        statistik.addChangeListener(new Kommentierung(spielplatz));
        fenster.zeige(eigenschaften, menue, spielplatz); // Zeige vor der automatischen Fortsetzung das Fenster!

        // Behandle Fortsetzung zuletzt, wenn die Infrastruktur aufgebaut ist.
        if (fortsetzung != null) {
            fortsetzung.addPropertyChangeListener("Speichern", menue);
            fortsetzung.verbinde(this);
            fortsetzung.versuche();
        }

    }

    /**
     * Betreibt ein neues Turnier gem eines Plans.
     *
     * @param plan
     *            Der Plan, nach dem das neue Turnier betrieben werden soll.
     */
    public void betreibe(Plan plan) {
        betreibe(new Turnier(teilnehmer(eigenschaften, plan.namen), plan.beginner, plan.variante, plan.maximalanzahl,
                plan.erster), null);
    }

    /**
     * Betreibt ber ein Netzwerk ein neues Turnier gem eines Plans.
     *
     * @param plan
     *            Der Plan, nach dem das neue Turnier betrieben werden soll.
     * @param vermittlung
     *            Vermittlung zwischen den Austragungsorten; null, falls nicht ber Netzwerk betrieben.
     */
    public void betreibe(Plan plan, Vermittlung vermittlung) {
        Turnier kandidat = new Turnier(teilnehmer(eigenschaften, plan.namen), plan.beginner, plan.variante,
                plan.maximalanzahl, plan.erster);
        menue.propertyChange(new PropertyChangeEvent(this, "Netzwerk", false, true));
        vermittlung.verknuepfe(kandidat.wuerfelsatz(), kandidat);
        kandidat.addChangeListener(vermittlung);
        betreibe(kandidat, vermittlung);
    }

    /**
     * Betreibt ein Turnier.
     *
     * @param kandidat
     *            Das Turnier, entweder schon vorliegend oder vor Aufruf neu erzeugt.
     */
    public void betreibe(Turnier kandidat) {
        betreibe(kandidat, null);
    }

    /**
     * Erzeugt eine Vermittlung.
     *
     * @param eroeffnung
     *            Die Schnittstelle zur Erffnung eines Turnierbetriebs.
     * @return Instanz einer Vermittlung.
     */
    public Vermittlung vermittlung(Eroeffnung eroeffnung) {
        return new Vermittlung(eroeffnung, aufsicht);
    }

    /**
     * Betreibt ein Turnier.
     *
     * @param kandidat
     *            Das Turnier, entweder schon vorliegend oder vor Aufruf neu erzeugt.
     * @param vermittlung
     *            Vermittlung zwischen den Austragungsorten; null, falls nicht ber Netzwerk betrieben.
     */
    private void betreibe(Turnier kandidat, Vermittlung vermittlung) {
        if (turnier != null) {
            vorgaengerspielerzahl = turnier.teilnehmer().size();
            turnier.beende();
        }
        turnier = kandidat;
        turnier.setzeFort(statistik);
        turnier.addChangeListener(spielplatz);
        turnier.addChangeListener(menue);
        turnier.addChangeListener(ende);
        turnier.auswertung().addChangeListener(spielplatz);
        for (int w = 0; w < Turnier.WUERFELSATZGROESSE; w++) {
            turnier.wuerfel(w).addChangeListener(spielplatz);
            turnier.wuerfel(w).addChangeListener(aufsicht.wuerfelbeobachter(w));
        }
        turnier.addChangeListener(fortsetzung);
        turnier.addChangeListener(aufsicht);
        if (fenster != null) {
            fenster.setTitle("JaFuffy " + "(" + turnier.auswertung() + ", " + turnier.beginner() + ")");
            if (vorgaengerspielerzahl != turnier.teilnehmer().size()) {
                SwingUtilities.invokeLater(() -> fenster.pack());
            }
        }
        aufsicht.verknuepfe(vermittlung);
        turnier.starte();
    }

    /**
     * Erzeugt eine Liste der teilnehmenden Spieler anhand der Namensliste aus diesem Plan.
     *
     * @param eigenschaften
     *            Allgemeine Programm- und Turniereigenschaften.
     * @param namen
     *            Die Namen aller Turnierteilnehmer, gegebenenfalls ber das ganze Netzwerk.
     * @return Liste, bestehend aus allen teilnehmenden Spielern, abgeleitet von der Namensliste.
     */
    ArrayList<Spieler> teilnehmer(Eigenschaften eigenschaften, ArrayList<Name> namen) {
        ArrayList<Spieler> teilnehmer = new ArrayList<>(Spieler.SPIELER);
        for (Name name : namen) {
            try {
                teilnehmer.add(name.spieler(eigenschaften));
            } catch (InstantiationException | IllegalAccessException | NoSuchMethodException | IllegalArgumentException
                    | InvocationTargetException ausnahme) {
                ausnahme.printStackTrace();
                throw new RuntimeException("Programmierfehler: Unbekannter Spielertyp.");
            }
        }
        return teilnehmer;
    }
}
