/******************************************************************************
 ** $Id: Turnier.java 2619 2021-02-21 17:31:16Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.ListIterator;
import java.util.Random;

import javax.swing.SwingUtilities;

import jafuffy.bedienung.auswahl.Auswahl;
import jafuffy.logik.analyse.Analyse;
import jafuffy.logik.auswertung.Auswertung;
import jafuffy.logik.ereignis.Ablauf;
import jafuffy.logik.ereignis.Aenderungen;
import jafuffy.netzwerk.Vermittlung.Nachfuehrung;

/** Ablaufsteuerung eines Turniers. */
public class Turnier extends Aenderungen<Ablauf> implements Nachfuehrung, ActionListener, Serializable {

    /** Die Fortschrittsdaten des Turniers. */
    private static class Fortschritt implements Serializable {

        /** Zur Serialisierung. */
        private static final long serialVersionUID = 8968279815699993227L;

        /** Spieler, der gerade dran ist. Der Wert "null" bedeutet, dass das Turnier beendet ist. */
        private Spieler aktiver;
        /** Anzahl der gesetzten Eintrge, wobei eine Runde beendet sein muss. */
        private int kategorienanzahl = 0;
        /**
         * Index des Spielers pro Runde, der gerade an der Reihe ist. Zhlung fngt bei Null mit dem Spieler an, welcher
         * die Runde erffnet. Der Index ist nicht identisch mit dem Index des Containers, welcher die Spieler enthlt.
         */
        private int spielerindex = 0;
        /** Anzahl der vollendeten Spiele, entspricht dem Index des laufenden Spieles im Turnier. */
        private int spielindex = 0;

        /**
         * Nimmt einen Schritt zurck.
         *
         * @param schritt
         *            Der Schritt, zu dem die Rcknahme erfolgt.
         * @param spieler
         *            Der Stand des Spielers, zu dem die Rcknahme erfolgt.
         * @param kategorie
         *            Eintrag in der Tabelle, der zurckgenommen wird.
         * @param auswertung
         *            Die Auswertung zur Abschaltung der Tipps.
         */
        void annulliere(Fortschritt schritt, Spieler spieler, Kategorie kategorie, Auswertung auswertung) {
            aktiver.beende();
            aktiver = schritt.aktiver;
            spielerindex = schritt.spielerindex;
            spielindex = schritt.spielindex;
            kategorienanzahl = schritt.kategorienanzahl;
            aktiver.annulliere(spieler, kategorie, auswertung.analyse());
            auswertung.reagiereAufWurf(aktiver);
            auswertung.aktiviereTipps(false);
        }

        /**
         * Setze Eintrag mit gegebenen Wert und merke den Schritt als Zwischenschritt fr eine sptere Rcknahme.
         *
         * @param schritt
         *            Fortschritt in diesem Schritt zwischenspeichern.
         * @param kategorie
         *            Der fr den aktiven Spieler momentan zu setzende Eintrag.
         * @param analyse
         *            Zur Analyse, welcher Wert schlussendlich gesetzt wird.
         * @return Kopie des Spielers im Zwischenschritt.
         */
        Spieler setze(Fortschritt schritt, Kategorie kategorie, Analyse analyse) {
            schritt.aktiver = aktiver;
            schritt.spielerindex = spielerindex;
            schritt.spielindex = spielindex;
            schritt.kategorienanzahl = kategorienanzahl;
            Spieler spieler = aktiver.clone();
            aktiver.belege(kategorie, analyse);
            aktiver.beende();
            return spieler;
        }
    }

    /** Die Anzahl der Wrfel, die fr das Spiel verwendet werden. */
    public static final int WUERFELSATZGROESSE = 5;

    /** Die Anzahl der maximal mglichen Wurfversuche pro Durchgang. */
    public static final int WURFVERSUCHANZAHL = 3;

    /** Anzahl der mglichen Eintrge auf Gewinnkarte . */
    private static final int KATEGORIENANZAHL = 13;
    /** Zur Serialisierung. */
    private static final long serialVersionUID = -8157448193770055452L;
    /** Zur zuflligen Ermittlung des erffnenden Spielers. */
    private static final Random ZUFALL = new Random();

    /** Die fr dieses Turnier gltige Auswertungsregel. */
    private final Auswertung auswertung;
    /** Hlt fest, wer nach Beendigung eines Spiels das nchste Spiel beginnt. */
    private final Beginner beginner;
    /** Der aktuelle Fortschritt im Turnier. */
    private final Fortschritt fortschritt = new Fortschritt();
    /** Der zwischengespeicherte Fortschritt im Turnier fr eine sptere Rcknahme. */
    private final Fortschritt fortschrittmerker = new Fortschritt();
    /** Hiermit wird der letzte gesetzte Eintrag gemerkt fr eine etwaige Rcknahme des Zuges. */
    private Kategorie kategoriemerker;
    /** Maximale Anzahl der Spiele im Turnier. */
    private final int maximalanzahl;
    /**
     * Punktestand. Muss wiederhergestellt werden, da das Modell Listener enthlt, die bei jedem Programmlauf neu
     * erstellt werden.
     */
    private transient Punkte punkte;
    /** Hiermit wird der Stand des letzten Spielers gemerkt fr eine etwaige Rcknahme des Zuges. */
    private Spieler standmerker;

    /** Statistik. Muss wiederhergestellt werden, da bei jedem Programmlauf aktualisiert. */
    private transient Statistik statistik;
    /** Alle an dem Turnier teilnehmenden Mitspieler. */
    private final ArrayList<Spieler> teilnehmer;
    /** Der aktuelle Turnierstand summiert ber alle abgeschlossenen Spiele, fr jeden Mitspieler eine Punktezahl. */
    private ArrayList<Integer> turnierstaende;
    /** Spielregelvariante, die fr das Turnier benutzt wird. */
    private final Variante variante;

    /** Der Turnierverlauf, alle Spieler pro Turnier werden gespeichert, was die Erstellung eines Reports ermglicht. */
    private ArrayList<ArrayList<Integer>> verlaeufe;
    /** Wrfel im Spiel. */
    private final Wuerfel[] wuerfelsatz;

    /**
     * Konstruktor, der von Initialisierungsfunktion gefolgt werden muss.
     *
     * @param teilnehmer
     *            Die Namen aller an dem Turnier teilnehmenden Spieler.
     * @param beginner
     *            Hlt fest, wer nach Beendigung eines Spiels das nchste Spiel beginnt.
     * @param variante
     *            Spielregelvariante, die fr das Turnier benutzt wird.
     * @param maximalanzahl
     *            Maximale Anzahl der Spiele im Turnier.
     * @param erster
     *            Spieler, welcher das Turnier erffnet.
     */
    public Turnier(ArrayList<Spieler> teilnehmer, Beginner beginner, Variante variante, int maximalanzahl, int erster) {
        this.teilnehmer = teilnehmer;
        this.beginner = beginner;
        this.variante = variante;
        this.maximalanzahl = maximalanzahl;
        wuerfelsatz = new Wuerfel[Turnier.WUERFELSATZGROESSE];
        for (int i = 0; i < Turnier.WUERFELSATZGROESSE; i++) {
            wuerfelsatz[i] = new Wuerfel(i + 1);
        }
        auswertung = variante.auswertung(wuerfelsatz);
        for (Spieler spieler : teilnehmer) {
            spieler.initialisiere();
        }
        fortschritt.aktiver = teilnehmer.get(erster);
        fortschritt.aktiver.aktiviere();
    }

    @Override
    public void actionPerformed(ActionEvent ereignis) {
        String kommando = ereignis.getActionCommand();
        if (kommando.equals("Gewuerfelt")) {
            leere();
            reagiere();
        } else if (kommando.equals("Rueckgaengig")) {
            annulliere();
        } else if (kommando.equals("Vorschlagen")) {
            auswertung.erstelleVorschlag();
        } else if (kommando.equals("Beenden")) {
            beende();
        }
    }

    /** Liefert den momentan aktiven Spieler. */
    public Spieler aktiver() {
        return fortschritt.aktiver;
    }

    /** Liefert den gemerkten derzeit aktiven Spieler. */
    public Spieler aktivermerker() {
        return fortschrittmerker.aktiver;
    }

    /** Liefert das Objekt, welches die Auswahl und Bearbeitung einer Zelle handhabt. */
    public Auswahl auswahl() {
        return variante.auswahl(this);
    }

    /** Liefert das Objekt, welches die Auswertung von Wrfen durchfhrt. */
    public Auswertung auswertung() {
        return auswertung;
    }

    /**
     * Erledigt Abschlussarbeiten, im Strungsfall Abbrucharbeiten, sichert gegebenenfalls den Turnierstand und
     * veranlasst die Anpassung von Darstellungen.
     */
    public void beende() {
        if (fortschritt.aktiver != null) {
            auswertung.aktiviereTipps(false);
            fireStateChanged(
                    fortschritt.spielindex == maximalanzahl && maximalanzahl > 0 ? Ablauf.ENDE : Ablauf.ABBRUCH);
            fortschritt.aktiver.beende();
            fortschritt.aktiver = null;
        }
    }

    /** Gibt an, ob das Turnier beendet ist. */
    public boolean beendet() {
        return fortschritt.aktiver == null;
    }

    /**
     * Belegt einen gegebenen Eintrag in der Tabelle mit dem erwrfelten Resultat.
     *
     * @param kategorie
     *            Der Eintrag, welcher belegt werden soll.
     */
    public void belege(Kategorie kategorie) {
        fortschritt.aktiver.belege(kategorie, auswertung.analyse());
    }

    /** Liefert eine fr das Sichern von Turnieren geeignete Turnierbeschreibung. */
    public String beschreibung() {
        String s = new String("Mitspieler: ");
        ListIterator<Spieler> iterator = teilnehmer.listIterator();
        while (iterator.hasNext()) {
            s += iterator.next().anzeigename();
            if (iterator.hasNext()) {
                s += ", ";
            } else {
                s += ".";
            }
        }
        s += "\n" + auswertung.toString() + ". " + (teilnehmer.size() > 1 ? beginner + ". " : "");
        s += maximalanzahl == 0 ? "Beliebig viele Spiele."
                : "Im " + (fortschritt.spielindex + 1) + ". von " + maximalanzahl
                        + (maximalanzahl != 1 ? " Spielen" : " Spiel") + ".";
        return s;
    }

    /** Liefert die gemerkte zuletzt gesetzte Kategorie. */
    public Kategorie kategoriemerker() {
        return kategoriemerker;
    }

    /** Liefert maximale Anzahl der Spiele pro Turnier, wobei Null Unbeschrnktheit bedeutet. */
    public int maximalanzahl() {
        return maximalanzahl;
    }

    /** Legt eine Pause ein, so dass keine Aktionen mehr entgegen genommen werden. */
    public void pausiere() {
        fireStateChanged(Ablauf.PAUSE);
    }

    /** Liefert das Punktemodell. */
    public Punkte punkte() {
        return punkte;
    }

    @Override
    public void reagiere() {
        fireStateChanged(Ablauf.LEERUNG);
        fortschritt.aktiver.reagiere();
        auswertung.reagiereAufWurf(fortschritt.aktiver);
        auswertung.aktiviereTipps(false);
        fireStateChanged(Ablauf.GEWORFEN);
    }

    /** Gibt an, ob der letzte Setzvorgang zurckgenommen werden kann. */
    public boolean ruecknahmefaehig() {
        return fortschrittmerker.aktiver.real();
    }

    @Override
    public void setze(Kategorie kategorie) {
        standmerker = fortschritt.setze(fortschrittmerker, kategorie, auswertung.analyse());
        kategoriemerker = kategorie;
        punkte.benachrichtige(fortschritt.aktiver.index());
        beendeSetzen();
    }

    /**
     * Weiter mit dem Turnier nach Turnierpause. Erforderlich wegen Serialisierung, bringt Turniersteuerung und
     * Wurfauswertung auf aktuellen Stand.
     *
     * @param statistik
     *            Aktuelles Statistikobjekt, das vom Turnier verwendet und gepflegt werden soll.
     */
    public void setzeFort(Statistik statistik) {
        this.statistik = statistik;
        auswertung.reagiereAufWurf(fortschritt.aktiver);
        punkte = new Punkte(teilnehmer, auswertung);
        if (verlaeufe == null) {
            verlaeufe = new ArrayList<>(teilnehmer.size());
            turnierstaende = new ArrayList<>(teilnehmer.size());
            for (Spieler spieler : teilnehmer) {
                verlaeufe.add(spieler.index(), new ArrayList<>());
                verlaeufe.get(spieler.index()).add(0);
                turnierstaende.add(spieler.index(), 0);
            }
        }
        statistik.verfolge(variante, maximalanzahl, teilnehmer, auswertung.mittelwert(), auswertung.abweichung(),
                verlaeufe, turnierstaende);
    }

    /** Liefert die Nummer des laufenden Spiels. */
    public int spielnummer() {
        return fortschritt.spielindex + 1;
    }

    /** Startet dieses Turnier oder setzt es fort. */
    public void starte() {
        fireStateChanged(Ablauf.START);
    }

    /** Liefert aller Teilnehmer diese Turniers. */
    public ArrayList<Spieler> teilnehmer() {
        return teilnehmer;
    }

    /** Gibt an, ob fr einen Wurfergebnis ein Vorschlag gemacht werden kann. */
    public boolean vorschlagbar() {
        return fortschritt.aktiver.gewuerfelt();
    }

    /**
     * Liefert den Wrfel zu einem Index.
     *
     * @param i
     *            Index des Wrfels
     * @return Wrfel
     */
    public Wuerfel wuerfel(int i) {
        return wuerfelsatz[i];
    }

    /** Liefert den Wrfelsatz vom Turnier. */
    public Wuerfel[] wuerfelsatz() {
        return wuerfelsatz;
    }

    /** Nimmt den Eintrag eines Spielers in die Punktetabelle zurck. */
    private void annulliere() {
        if (fortschritt.kategorienanzahl != fortschrittmerker.kategorienanzahl) {
            statistik.annulliere();
        }
        fortschritt.annulliere(fortschrittmerker, standmerker, kategoriemerker, auswertung);
        for (Wuerfel wuerfel : wuerfelsatz) {
            wuerfel.darstellungsmodell().setSelected(false);
        }
        fireStateChanged(Ablauf.RUECKGAENGIG);
    }

    /** Runde beenden. */
    private void beendeRunde() {
        fortschritt.kategorienanzahl++;
        fortschritt.spielerindex = 0;
        statistik.beendeRunde();
    }

    /**
     * Spieler hat gesetzt, der nchste ist dran. Es werden keine for/while-Schleifen benutzt, da diese Methode bei
     * jedem Setz-Ereignis angesprungen wird.
     */
    private void beendeSetzen() {
        fortschritt.aktiver = naechster();
        fortschritt.aktiver.aktiviere();
        fortschritt.spielerindex++;
        if (fortschritt.spielerindex >= teilnehmer.size()) {
            beendeRunde();
            if (fortschritt.kategorienanzahl >= KATEGORIENANZAHL) {
                fortschritt.aktiver.beende(); // Nach Ende eines Spiels soll keine Tabellenspalte markiert sein
                auswertung.aktiviereTipps(false);
                fireStateChanged(Ablauf.WECHSEL);
                SwingUtilities.invokeLater(() -> {
                    fireStateChanged(Ablauf.RESULTAT);
                    beendeSpiel();
                    uebergebe();
                });
                return;
            }
        }
        uebergebe();
    }

    /** Spiel beenden, nchsten Spielerffner bestimmen. */
    private void beendeSpiel() {
        Spieler erster = eroeffner(); // Berechnungen erfordern Aufruf vor der Initialisierung der Spieler
        statistik.beendeSpiel();
        for (Spieler spieler : teilnehmer) {
            spieler.initialisiere();
        }
        fortschritt.aktiver = erster;
        fortschritt.aktiver.aktiviere();
        fortschritt.spielindex++;
        fortschritt.kategorienanzahl = 0;
    }

    /**
     * Bestimmt den Spieler, welcher das nchste Spiel im Turnier erffnet.
     *
     * @return Erffnender Spieler.
     */
    private Spieler eroeffner() {
        Spieler verlierer = teilnehmer.get(0);
        Spieler gewinner = verlierer;
        for (Spieler spieler : teilnehmer) {
            if (spieler.endsumme() < verlierer.endsumme()) {
                verlierer = spieler;
            }
            if (spieler.endsumme() > gewinner.endsumme()) {
                gewinner = spieler;
            }
        }
        Spieler eroeffner;
        switch (beginner) {
        case VERLIERER:
            eroeffner = verlierer;
            break;
        case GEWINNER:
            eroeffner = gewinner;
            break;
        case REIHUM:
            eroeffner = naechster();
            break;
        case ZUFALL:
            eroeffner = teilnehmer.get(ZUFALL.nextInt(teilnehmer.size()));
            break;
        case DERSELBE:
            eroeffner = fortschritt.aktiver;
            break;
        default:
            eroeffner = null; // Fataler Fehler, fhrt zum Abbruch. Auftreten weist auf Programmierfehler hin.
            break;
        }
        return eroeffner;
    }

    /** Leert den Wrfelbecher mit den darinliegenden Wrfeln, so dass ein Wrfelergebnis fr diese Wrfel vorliegt. */
    private void leere() {
        for (Wuerfel wuerfel : wuerfelsatz) {
            if (wuerfel.darstellungsmodell().isSelected()) {
                wuerfel.wirf();
            }
        }
    }

    /**
     * Bestimmt den Spieler, der als nchstes reihum drankommt.
     *
     * @return Nchster Spieler, der an der Reihe ist.
     */
    private Spieler naechster() {
        return teilnehmer.get((fortschritt.aktiver.index() + 1) % teilnehmer.size());
    }

    /** bergibt die Wrfel an den nchsten Teilnehmer durch Senden von Ereignissen. */
    private void uebergebe() {
        if (maximalanzahl > 0 && fortschritt.spielindex == maximalanzahl) {
            beende();
        } else {
            if (fortschritt.spielindex == fortschrittmerker.spielindex) {
                auswertung.aktiviereTipps(false);
                fireStateChanged(Ablauf.FERTIG);
            } else {
                fireStateChanged(Ablauf.SPIEL);
            }
        }
    }

    /** Liefert den Beginnmodus dieses Turniers. */
    Beginner beginner() {
        return beginner;
    }

    /**
     * Liefert den Spieler zu einem gegebenen Index.
     *
     * @param i
     *            Index
     * @return Spieler
     */
    Spieler spieler(int i) {
        return teilnehmer.get(i);
    }

}
