/******************************************************************************
 ** $Id: Klassisch.java 2533 2021-01-03 13:53:57Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.analyse;

import java.util.ArrayList;

import jafuffy.logik.Kategorie;
import jafuffy.logik.Spieler;
import jafuffy.logik.Turnier;
import jafuffy.logik.Wuerfel;

/** Erstellt Analysen und berechnet Prognosen fr die klassische Variante der Spielregeln. */
public class Klassisch extends Analyse {

    /**
     * Alle Eintrge fr die klassische Variante der Spielregeln als Tabelle zusammengefasst.
     */
    protected class Eintragungen extends Analyse.Eintragungen {
        /** Erlaubt Serialisierung. */
        private static final long serialVersionUID = -7416744679857533889L;

        /** Konstruktion der Tabelle. */
        Eintragungen() {
            put(Kategorie.DREIERPASCH, new Dreierpasch());
            put(Kategorie.VIERERPASCH, new Viererpasch());
            put(Kategorie.FULLHOUSE, new FullHouse());
            put(Kategorie.KLEINESTRASSE, new KleineStrasse());
            put(Kategorie.GROSSESTRASSE, new GrosseStrasse());
            put(Kategorie.JAFUFFY, new JaFuffy());
        }
    }

    /** Kleiner Pasch. */
    class Dreierpasch extends Pasch {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = 1347995442616468479L;
        /** Anzahl der gleichen Wrfel fr einen kleinen Pasch. */
        private static final int PASCHGROESSE = 3;

        /**
         * Konstruktor.
         */
        Dreierpasch() {
            super(PASCHGROESSE);
        }

        @Override
        protected double augensumme(int a, int r, int k) {
            if (k != 2) {
                throw new IllegalArgumentException("Falsche Mehrlingslnge");
            }
            if (r > 0) {
                double augensumme;
                augensumme = 4 * (a + 4) * augensumme(a, r - 1, 2) + 66 * Kombinatorik.augenzahl(r)
                        - 2 * Kombinatorik.augenzahl(a, r - 1) + 90 * a + 21;
                for (int alpha = a + 1; alpha <= 6; alpha++) {
                    augensumme += 4 * augensumme(alpha, r - 1, 2);
                }
                augensumme *= 3. / 216.;
                return augensumme;
            } else {
                return 0;
            }
        }

        @Override
        protected boolean pasch() {
            return paschkarte.get(Paschaufzaehlung.DREIERPASCH) || paschkarte.get(Paschaufzaehlung.VIERERPASCH)
                    || paschkarte.get(Paschaufzaehlung.FUENFERPASCH);
        }

    }

    /** Groer Pasch. */
    class Viererpasch extends Pasch {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -3378721831288162486L;
        /** Anzahl der gleichen Wrfel fr einen groen Pasch. */
        private static final int PASCHGROESSE = 4;

        /**
         * Konstruktor.
         */
        Viererpasch() {
            super(PASCHGROESSE);
        }

        @Override
        protected double augensumme(int a, int r, int k) {
            switch (k) {
            case 2:
                return augensumme2(a, r);
            case 3:
                return augensumme3(a, r);
            default:
                throw new IllegalArgumentException("Falsche Mehrlingslnge");
            }
        }

        /**
         * @param a
         *            Fest vorgegebene Augenzahl fr einen Zwilling, wobei bei gleich langen Sequenzen diejenige mit der
         *            greren Augenzahl zu whlen ist.
         * @param r
         *            Anzahl der Restwrfe.
         */
        protected double augensumme2(int a, int r) {
            if (r > 0) {
                double augensumme;
                augensumme = (48 + 12 * a) * augensumme2(a, r - 1) + 75 * augensumme3(a, r - 1)
                        + 18 * Kombinatorik.augenzahl(r) - 2 * Kombinatorik.augenzahl(a, r - 1) + 64 * a;
                for (int alpha = 0; alpha < a; alpha++) {
                    augensumme += augensumme3(alpha, r - 1);
                }
                for (int alpha = a + 1; alpha <= 6; alpha++) {
                    augensumme += 12 * augensumme2(alpha, r - 1);
                    augensumme += augensumme3(alpha, r - 1);
                }
                augensumme /= 216;
                return augensumme;
            } else {
                return 0;
            }
        }

        /**
         * @param a
         *            Fest vorgegebene Augenzahl fr einen Drilling, wobei bei gleich langen Sequenzen diejenige mit der
         *            greren Augenzahl zu whlen ist.
         * @param r
         *            Anzahl der Restwrfe.
         */
        protected double augensumme3(int a, int r) {
            if (r > 0) {
                double augensumme;
                augensumme = 25 * augensumme3(a, r - 1) + 44 * a + 12 * Kombinatorik.augenzahl(r)
                        - Kombinatorik.augenzahl(a, r - 1);
                augensumme /= 36;
                return augensumme;
            } else {
                return 0;
            }
        }

        @Override
        protected boolean pasch() {
            return paschkarte.get(Paschaufzaehlung.VIERERPASCH) || paschkarte.get(Paschaufzaehlung.FUENFERPASCH);
        }

    }

    /** Full House. */
    class FullHouse extends Unten {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -1069152277729232203L;
        /** Punktzahlen (Full House). */
        static final int PUNKTE = 25;
        /** Wahrscheinlichkeit fr 0, 1, ..., 5 fehlende Wrfel auf Full House in einem Wurfergebnis mit allen Wrfeln. */
        private final double[] restwahrscheinlichkeit = { 17. / 432., 25. / 108., 25. / 144., 25. / 54., 0., 5. / 54. };
        /** Anzahl der Wrfel, die zum Full House fehlen. */
        private int fehlend;

        /**
         * Konstruktor.
         */
        FullHouse() {
            startmittel = prognose(Turnier.WUERFELSATZGROESSE, Turnier.WURFVERSUCHANZAHL);
        }

        /** @return Liefert zurck, ob ein Full House vorliegt, wobei ein JaFuffy als Full House betrachtet wird. */
        private boolean istFullHouse() {
            return paschkarte.get(Paschaufzaehlung.ZWEIERPASCH) && paschkarte.get(Paschaufzaehlung.DREIERPASCH)
                    || paschkarte.get(Paschaufzaehlung.FUENFERPASCH);
        }

        /**
         * @param k
         *            Anzahl der zu Full House fehlenden Wrfel
         * @param r
         *            Anzahl der verbleibenden Restwrfe (0, ..., 3)
         * @return mittlere erreichbare Punktzahl
         */
        private double prognose(int k, int r) {
            return PUNKTE * restwahrscheinlichkeit(k, r);
        }

        /**
         * @param k
         *            Anzahl der zu Full House fehlenden Wrfel
         * @param r
         *            Anzahl der verbleibenden Restwrfe (0, ..., 3)
         * @return Wahrscheinlichkeit der Ergnzung zu Full House
         */
        private double restwahrscheinlichkeit(int k, int r) {
            switch (k) {
            case 0:
                return 1;
            case 1:
                return 1 - Math.pow(2. / 3., r);
            case 2:
                return 1 - Math.pow(Kombinatorik.AUSLASSUNGSWAHRSCHEINLICHKEIT, r);
            case 3:
                switch (r) {
                case 1:
                    return 7. / 72.;
                case 2:
                    return 89. / 324.;
                default:
                    return 0;
                }
            case 5:
                if (r == 0) {
                    return 0;
                }
                double p = 0;
                for (int i = 0; i <= Turnier.WUERFELSATZGROESSE; i++) {
                    p += restwahrscheinlichkeit[i] * restwahrscheinlichkeit(i, r - 1);
                }
                return p;
            default:
                return 0;
            }
        }

        @Override
        protected double ergebnismittel(Spieler spieler) {
            return prognose(fehlend, spieler.rest());
        }

        @Override
        protected void bewerte(Spieler spieler) {
            wert = istFullHouse() ? PUNKTE : 0;
        }

        @Override
        protected void waehleWuerfel(Spieler spieler) {
            super.waehleWuerfel(spieler);
            if (istFullHouse()) {
                // Full House liegt vor
                fehlend = 0;
            } else if (paschkarte.get(Paschaufzaehlung.DREIERPASCH) || paschkarte.get(Paschaufzaehlung.VIERERPASCH)) {
                // mindestens drei gleiche Augenzahlen liegen vor, behalte Drilling
                int zaehler = 0;
                for (Wuerfel wuerfel : wuerfelsatz) {
                    neuwurfkarte.put(wuerfel,
                            wuerfel.vorkommen(augenhaeufigkeiten) < Paschaufzaehlung.DREIERPASCH.groesse()
                                    || zaehler == Paschaufzaehlung.DREIERPASCH.groesse());
                    if (wuerfel.vorkommen(augenhaeufigkeiten) >= Paschaufzaehlung.DREIERPASCH.groesse()) {
                        zaehler++;
                    }
                }
                fehlend = 2;
            } else if (paschkarte.get(Paschaufzaehlung.ZWEIERPASCH)) {
                // ein Zwilling oder aber zwei Zwillinge mit unterschiedlicher Augenzahl
                fehlend = 0;
                for (int haeufigkeit : augenhaeufigkeiten) {
                    if (haeufigkeit == Paschaufzaehlung.ZWEIERPASCH.groesse()) {
                        if (fehlend == 0) { // erstes Paar gefunden
                            fehlend = 3;
                        } else { // es gibt ein zweites Paar
                            fehlend = 1;
                        }
                    }
                }
                for (Wuerfel wuerfel : wuerfelsatz) {
                    neuwurfkarte.put(wuerfel,
                            wuerfel.vorkommen(augenhaeufigkeiten) < Paschaufzaehlung.ZWEIERPASCH.groesse());
                }
            } else {
                // alle Wrfel verschieden
                nominiereNeuwurf(true);
                fehlend = Turnier.WUERFELSATZGROESSE;
            }
        }
    }

    /** Kleine Strae. */
    public class KleineStrasse extends Unten {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = 3736058022544823661L;
        /** Die Wahrscheinlichkeit der vorliegenden Strategie in drei Wrfen eine kleine Strae zu wrfeln. */
        private static final double P_TOT = 0.6025279;
        /** Punktzahlen (kleine Strae). */
        public static final int PUNKTE = 30;
        /** Anzahl der auen liegenden gnstigen Augen (1 oder 6). */
        private int aussen;
        /** Anzahl der innen liegenden gnstigen Augen (2 oder 5). */
        private int innen;
        /** Anzahl der mittig liegenden gnstigen Augen (3 oder 4). */
        private int mitte;

        /**
         * Konstruktor.
         */
        KleineStrasse() {
            startmittel = P_TOT * PUNKTE;
        }

        /**
         * Whle Wrfel so aus, dass ein Straenteilpaar behalten wird (sofern die Paarteile im Wurfergebnis enthalten sind).
         *
         * @param paar
         *            Das zu beibehaltende Straenteilpaar.
         */
        private void behalte(Strasse.Paar paar) {
            for (Strasse.Index element : paar.feld()) {
                if (augenhaeufigkeiten[element.ordinal()] > 0) { // Kommt Augenzahl im Wurfergebnis vor?
                    for (Wuerfel w : wuerfelsatz) {
                        if (w.augen() == element.augen()) {
                            neuwurfkarte.put(w, false);
                            break;
                        }
                    }
                }
            }
        }

        @Override
        protected void bewerte(Spieler spieler) {
            wert = Strasse.Index.istKleineStrasse(augenhaeufigkeiten) ? PUNKTE : 0;
        }

        @Override
        protected double ergebnismittel(Spieler spieler) {
            if (Strasse.Index.istKleineStrasse(augenhaeufigkeiten)) {
                return PUNKTE;
            } else if (spieler.rest() > 0) {
                return Strasse.KleinerIndex.TABELLE.get(new Strasse.KleinerIndex(aussen, innen, mitte, spieler.rest()))
                        * PUNKTE;
            } else {
                return 0;
            }
        }

        /**
         * Bereitet die Wrfelauswahl vor.
         *
         * @param fertig
         *            Bestimmt, ob bei Erreichen der kleine Strae mit dem Wrfeln aufgehrt werden soll.
         */
        protected void waehleWuerfel(boolean fertig) {
            nominiereNeuwurf(!fertig);
            if (!fertig) {
                aussen = 0;
                innen = Strasse.Paar.INNEN.vorkommen(augenhaeufigkeiten);
                mitte = Strasse.Paar.MITTE.vorkommen(augenhaeufigkeiten);
                int links = Strasse.Paar.LINKS.vorkommen(augenhaeufigkeiten);
                int rechts = Strasse.Paar.RECHTS.vorkommen(augenhaeufigkeiten);
                if (innen == 2) {
                    behalte(Strasse.Paar.INNEN);
                } else if (innen == 1 && links <= 1 && rechts <= 1) {
                    behalte(Strasse.Paar.INNEN);
                } else {
                    if (links > rechts) {
                        aussen = links - innen;
                        behalte(Strasse.Paar.LINKS);
                    } else if (links < rechts) {
                        aussen = rechts - innen;
                        behalte(Strasse.Paar.RECHTS);
                    } else {
                        aussen = links - innen;
                        behalte(ZUFALL.nextBoolean() ? Strasse.Paar.LINKS : Strasse.Paar.RECHTS);
                    }
                }
                behalte(Strasse.Paar.MITTE);
            }
        }

        @Override
        protected void waehleWuerfel(Spieler spieler) {
            super.waehleWuerfel(spieler);
            waehleWuerfel(false);
        }
    }

    /** Groe Strae. */
    public class GrosseStrasse extends Unten {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = 8292592862416663732L;
        /** Punktzahlen (groe Strae). */
        public static final int PUNKTE = 40;
        /** Anzahl der Wrfel je Augenzahl. */
        private final int[] kandidat = new int[Wuerfel.MAXIMALAUGENAUGENZAHL];
        /** Anzahl der fehlenden Wrfel zu einer groe Strae. */
        private int fehlend;

        /**
         * Konstruktor.
         */
        GrosseStrasse() {
            startmittel = PUNKTE * q(Turnier.WUERFELSATZGROESSE, Turnier.WURFVERSUCHANZAHL);
        }

        /**
         * Wahrscheinlichkeit zur groen Strae, falls 1 oder 6 schon erwrfelt wurde.
         *
         * @param n
         *            Anzahl der fehlenden Wrfel zur groen Strae, Null ist erlaubt.
         * @param r
         *            Anzahl der noch mglichen Restwrfe, Null ist erlaubt.
         * @return Wahrscheinlichkeit
         */
        private double p(int n, int r) {
            if (r > 1) {
                double summe = 0;
                for (int k = 0; k <= n; k++) {
                    summe += u(n, k) * p(n - k, r - 1);
                }
                return summe;
            } else if (r == 1) {
                return Kombinatorik.fakultaet(n) / Math.pow(Wuerfel.MAXIMALAUGENAUGENZAHL, n);
            } else {
                if (n > 0) {
                    return 0; // Strae kann nicht mehr erreicht werden.
                } else {
                    return 1; // Strae liegt schon vor.
                }
            }
        }

        /**
         * Wahrscheinlichkeit zur groen Strae, falls weder 1 noch 6 erwrfelt wurden.
         *
         * @param n
         *            Anzahl der fehlenden Wrfel zur groen Strae
         * @param r
         *            Anzahl der noch mglichen Restwrfe
         * @return Wahrscheinlichkeit
         */
        private double q(int n, int r) {
            if (r > 1) {
                double summe = 0;
                for (int k = 0; k <= n; k++) {
                    summe += w(n, k) * q(n - k, r - 1) + v(n, k) * p(n - k, r - 1);
                }
                return summe;
            } else if (r == 1) {
                return 2 * Kombinatorik.fakultaet(n) / Math.pow(Wuerfel.MAXIMALAUGENAUGENZAHL, n);
            } else {
                return 0; // Strae kann nicht erreicht werden, da sowohl 1 als auch 6 fehlen.
            }
        }

        /**
         * Wahrscheinlichkeit, genau k unterschiedliche Augen aus n vorgegebenen Augen mit n Wrfeln zu erwrfeln. Die
         * nicht vorgegebenen Augen drfen beliebig oft auftauchen.
         *
         * @param n
         *            Anzahl der Wrfel
         * @param k
         *            Anzahl der unterschiedlichen Augen
         * @return Wahrscheinlichkeit
         */
        private double u(int n, int k) {
            return Kombinatorik.strassenteilsequenz(n, Wuerfel.MAXIMALAUGENAUGENZAHL - n, k, n)
                    / Math.pow(Wuerfel.MAXIMALAUGENAUGENZAHL, n);
        }

        /**
         * Wahrscheinlichkeit, genau k unterschiedliche Augen aus n-1 erlaubten Augen mit n Wrfeln zu erwrfeln, wobei
         * zustzlich 1 oder 6 vorkommen muss. Alle anderen Augen drfen beliebig oft auftauchen.
         *
         * @param n
         *            Anzahl der Wrfel
         * @param k
         *            Anzahl der unterschiedlichen Augen
         * @return Wahrscheinlichkeit
         */
        private double v(int n, int k) {
            return k * Kombinatorik.fallend(n - 1, k - 1)
                    * (2 * Kombinatorik.strassenteilschablone(n, 5 - n, k)
                            + (k + 1) * Kombinatorik.strassenteilschablone(n, 5 - n, k + 1))
                    / Math.pow(Wuerfel.MAXIMALAUGENAUGENZAHL, n);
        }

        /**
         * Wahrscheinlichkeit, mit n Wrfeln genau k unterschiedliche Augen aus n erlaubten Augen zu erwrfeln, wobei
         * weder 1 noch 6 vorkommen darf. Die restlichen Wrfel drfen alle allen anderen Augen aufweisen, wieder
         * abgesehen von 1 und 6.
         *
         * @param n
         *            Anzahl der Wrfel
         * @param k
         *            Anzahl der unterschiedlichen Augen (ohne 1 und 6)
         * @return Wahrscheinlichkeit
         */
        private double w(int n, int k) {
            return Kombinatorik.strassenteilsequenz(n, Wuerfel.MAXIMALAUGENAUGENZAHL - 1 - n, k, n - 1)
                    / Math.pow(Wuerfel.MAXIMALAUGENAUGENZAHL, n);
        }

        @Override
        protected double ergebnismittel(Spieler spieler) {
            if (augenhaeufigkeiten[Strasse.Index.EINSER.ordinal()] > 0
                    || augenhaeufigkeiten[Strasse.Index.SECHSER.ordinal()] > 0) {
                return PUNKTE * p(fehlend, spieler.rest()); // 1 oder 6 wurde schon erwrfelt
            } else {
                return PUNKTE * q(fehlend, spieler.rest()); // 1 oder 6 noch zu erwrfeln
            }
        }

        @Override
        protected void bewerte(Spieler spieler) {
            if (Strasse.Index.istGrosseStrasse(augenhaeufigkeiten)) {
                wert = PUNKTE;
            } else {
                wert = 0;
            }
        }

        @Override
        protected void waehleWuerfel(Spieler spieler) {
            super.waehleWuerfel(spieler);
            System.arraycopy(augenhaeufigkeiten, 0, kandidat, 0, Wuerfel.MAXIMALAUGENAUGENZAHL);
            if (kandidat[Strasse.Index.EINSER.ordinal()] > 0 && kandidat[Strasse.Index.SECHSER.ordinal()] > 0) {
                // 1 und 6 im Wurfergebnis, zufllig 1 oder 6 ausschlieen
                if (ZUFALL.nextBoolean()) {
                    kandidat[Strasse.Index.EINSER.ordinal()] = 0; // 1 erneut wrfeln (1 virtuell aus Wurfergebnis entfernen)
                } else {
                    kandidat[Strasse.Index.SECHSER.ordinal()] = 0; // 6 erneut wrfeln (6 virtuell aus Wurfergebnis entfernen)
                }
            }
            fehlend = 0;
            for (Wuerfel wuerfel : wuerfelsatz) { // Wrfel auswhlen
                int i = wuerfel.augen() - 1;
                neuwurfkarte.put(wuerfel, kandidat[i] == 0);
                // weitere Wrfel mit selber Augenzahl ignorieren
                if (neuwurfkarte.get(wuerfel)) {
                    fehlend++;
                } else {
                    kandidat[i] = 0;
                }
            }
        }
    }

    /** JaFuffy. */
    class JaFuffy extends Unten {
        /** Zur Serialisierung. */
        private static final long serialVersionUID = -6921954073317063844L;
        /** Punktzahlen (JaFuffy). */
        static final int PUNKTE = 50;

        /**
         * Konstruktor.
         */
        JaFuffy() {
            startmittel = prognose(Turnier.WURFVERSUCHANZAHL, 0);
        }

        /**
         * @param r
         *            Verbleibende Anzahl der Wrfe.
         * @param max
         *            Maximale Anzahl gleicher Augen; Null ist erlaubt und meint den ersten von drei Wrfen.
         * @return Wahrscheinlichkeit zur Erzielung eines JaFuffy.
         */
        private double p(int r, int max) {
            if (r > 0) {
                double p = 0;
                if (max <= 1) {
                    for (int i = 1; i <= Turnier.WUERFELSATZGROESSE; i++) {
                        p += Kombinatorik.gleichenwahrscheinlichkeit(i) * p(r - 1, i);
                    }
                } else if (max == 2) {
                    p = (80. * Kombinatorik.fixgleichenwahrscheinlichkeit(2, r - 1)
                            + 15. * Kombinatorik.fixgleichenwahrscheinlichkeit(1, r - 1) + 1 + 120. * p(r - 1, 2))
                            / 216.;
                } else {
                    p = Kombinatorik.fixgleichenwahrscheinlichkeit(Turnier.WUERFELSATZGROESSE - max, r);
                }
                return p;
            } else {
                return max == Turnier.WUERFELSATZGROESSE ? 1 : 0;
            }
        }

        /**
         * @param r
         *            Verbleibende Anzahl der Wrfe.
         * @param max
         *            Maximale Anzahl gleicher Augen; Null ist erlaubt und meint den ersten von drei Wrfen.
         * @return Mittlere erreichbare Punktzahl.
         */
        private double prognose(int r, int max) {
            return p(r, max) * PUNKTE;
        }

        @Override
        protected double ergebnismittel(Spieler spieler) {
            return prognose(spieler.rest(), paschlaenge());
        }

        @Override
        protected void bewerte(Spieler spieler) {
            wert = paschkarte.get(Paschaufzaehlung.FUENFERPASCH) ? PUNKTE : 0;
        }

        @Override
        protected void waehleWuerfel(Spieler spieler) {
            super.waehleWuerfel(spieler);
            int max = paschlaenge(); // maximale Anzahl gleicher Augen
            int augen; // mit dieser Augenzahl wird JaFuffy versucht
            ArrayList<Integer> augenauswahl = new ArrayList<>();
            // suche Augenzahl mit maximalem Vorkommen heraus
            for (int ind = 0; ind < Wuerfel.MAXIMALAUGENAUGENZAHL; ind++) {
                if (augenhaeufigkeiten[ind] == max) {
                    augenauswahl.add(Integer.valueOf(ind + 1));
                }
            }
            // bei Mehrdeutigkeiten zufllige Auswahl der Augenzahl
            augen = augenauswahl.get(ZUFALL.nextInt(augenauswahl.size())).intValue();
            // Wrfel auswhlen
            for (Wuerfel w : wuerfelsatz) {
                // Falls alle Wrfel verschieden alle neu wrfeln, ansonsten mit eben ausgewhlter Augenzahl.
                neuwurfkarte.put(w, max == 1 || augen != w.augen());
            }
        }
    }

    /** Erlaubt Serialisierung. */
    private static final long serialVersionUID = -3852043350163717864L;

    /**
     * Konstruktor.
     *
     * @param wuerfel
     *            Wrfel, mit denen das Turnier durchgefhrt wird.
     */
    public Klassisch(Wuerfel[] wuerfel) {
        super(wuerfel);
    }

    @Override
    protected Analyse.Eintragungen eintragungen() {
        return new Eintragungen();
    }

}
