/******************************************************************************
 ** $Id: Zettel.java 2619 2021-02-21 17:31:16Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.bedienung;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ContainerAdapter;
import java.awt.event.ContainerEvent;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jafuffy.Eigenschaften;
import jafuffy.bedienung.auswahl.Auswahl;
import jafuffy.logik.Kategorie;
import jafuffy.logik.Punkte;
import jafuffy.logik.Turnier;
import jafuffy.logik.Wuerfel;
import jafuffy.logik.ereignis.Ablauf;
import jafuffy.logik.ereignis.Umschlag;

/**
 * Auf dem Zettel werden die Punktstnde notiert. Zugehriges Modell in Klasse "Punkte", durchgereicht an Darstellung
 * "Tabelle".
 */
@SuppressWarnings("serial")
class Zettel extends JScrollPane implements ChangeListener {

    /** Darstellung des Zeilenkopfs fr Punkteintrge. */
    private class Zeilenkopf extends JLabel {

        /** Abstand zwischen Zeilenkopfdeckel und Inhalt. */
        private static final int DECKELABSTAND = 4;
        /** Abstand zwischen Zeilenkopfseite und Inhalt. */
        private static final int SEITENABSTAND = 2;

        /** Konstruktion. */
        Zeilenkopf() {
            setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEtchedBorder(),
                    BorderFactory.createEmptyBorder(SEITENABSTAND, DECKELABSTAND, SEITENABSTAND, DECKELABSTAND)));
            setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
            faerbe();
        }

        @Override
        public void updateUI() {
            super.updateUI();
            faerbe();
        }

        /** Frbt den Zeilenkopf ein. */
        protected void faerbe() {
            setForeground(Color.DARK_GRAY);
        }
    }

    /** Darstellung des Zeilenkopfs fr Endsumme. */
    private class ZeilenkopfEnde extends Zeilenkopf {
        @Override
        protected void faerbe() {
            setBackground(Zettel.this.getBackground().darker());
            setForeground(Color.BLACK);
        }
    }

    /** Darstellung des Zeilenkopfs fr Zwischensummen. */
    private class ZeilenkopfZwischen extends Zeilenkopf {
        @Override
        protected void faerbe() {
            Color bg1 = Zettel.this.getBackground();
            Color bg2 = bg1.darker();
            Color neu = new Color((bg1.getRed() + bg2.getRed()) / 2, (bg1.getGreen() + bg2.getGreen()) / 2,
                    (bg1.getBlue() + bg2.getBlue()) / 2);
            setBackground(neu);
            setForeground(Color.BLACK);
        }
    }

    /** Grafiken fr Wrfelchen. */
    static final ImageIcon[] WUERFEL = new ImageIcon[Wuerfel.MAXIMALAUGENAUGENZAHL];

    static {
        for (int i = 0; i <= Turnier.WUERFELSATZGROESSE; i++) {
            WUERFEL[i] = Grafik.icon("Wuerfelchen" + (i + 1) + ".gif");
        }
    }

    static {
        for (int i = 0; i <= Turnier.WUERFELSATZGROESSE; i++) {
            WUERFEL[i] = Grafik.icon("Wuerfelchen" + (i + 1) + ".gif");
        }
    }

    /** Die Eigenschaften von JaFuffy. */
    private final Eigenschaften eigenschaften;
    /** Zeilenkpfe. */
    private final JPanel zeilenkoepfe;
    /** Das Objekt zum Setzen des Wurfs. */
    private final Setzen setzen;
    /** Tabellenhhe, ermittelt aus Zeilenkpfen. */
    private int hoehe;

    /**
     * Konstruktor.
     *
     * @param eigenschaften
     *            Die Eigenschaften von JaFuffy.
     * @param setzen
     *            Das Objekt zum Setzen des Wurfs.
     */
    Zettel(Eigenschaften eigenschaften, Setzen setzen) {
        this.eigenschaften = eigenschaften;
        setCorner(UPPER_LEFT_CORNER, erzeugeEcke());
        zeilenkoepfe = erzeugeZeilenkoepfe();
        this.setzen = setzen;
        setzeIcons();
        setRowHeaderView(zeilenkoepfe);
        synchronisiereZeilenhoehen();
    }

    @Override
    public void stateChanged(ChangeEvent ce) {
        Tabelle tabelle = (Tabelle) getViewport().getView();
        if (Umschlag.adressiert(ce, Ablauf.class)) {
            Umschlag<Ablauf, Turnier> umschlag = Umschlag.ereignisbehaelter(ce);
            switch (umschlag.ereignis()) {
            case START:
                Auswahl auswahl = umschlag.quelle().auswahl();
                getViewport().setView(new Tabelle(eigenschaften, umschlag.quelle().punkte(), auswahl));
                setzen.addActionListener(auswahl);
                break;
            case GEWORFEN:
                ((Auswahl) tabelle.getDefaultEditor(Object.class)).reagiereAufWurf();
                break;
            case PAUSE:
            case WECHSEL:
                ((Punkte) tabelle.getModel()).auswaehlbar(false);
                break;
            case FERTIG:
            case RUECKGAENGIG:
            case RESULTAT:
                ((Punkte) tabelle.getModel()).auswaehlbar(true);
                break;
            case VORGESCHLAGEN:
                tabelle.getDefaultEditor(Object.class).cancelCellEditing();
                break;
            case ABBRUCH:
            case ENDE:
                setzen.removeActionListener(((Auswahl) tabelle.getDefaultEditor(Object.class)));
                break;
            default:
                break;
            }
        }
        if (tabelle != null) {
            tabelle.repaint();
        }
    }

    /** Aktualisiert die gewnschte Gre der Punktetabelle. */
    private void aktualisiereTabellengroesse() {
        Tabelle tabelle = (Tabelle) getViewport().getView();
        int breite = tabelle.breite();
        hoehe = zeilenkoepfe.getPreferredSize().height;
        tabelle.setPreferredScrollableViewportSize(new Dimension(breite, hoehe));
    }

    /**
     * Aktualisiert die Zeilenhhen der Punktetabelle, so dass sie mit der tatschlichen Tabellengre bereinstimmen.
     */
    private void aktualisiereZeilenhoehe() {
        Tabelle tabelle = (Tabelle) getViewport().getView();
        if (tabelle != null) {
            for (int zeile = 0; zeile < tabelle.getRowCount(); zeile++) {
                tabelle.setRowHeight(zeile, zeilenkoepfe.getComponent(zeile).getHeight());
            }
        }
    }

    /**
     * @return Ecke.
     */
    private JLabel erzeugeEcke() {
        JLabel ecke = new JLabel("JaFuffy");
        ecke.setHorizontalAlignment(SwingConstants.CENTER);
        ecke.setToolTipText("Felderluterungen siehe Hilfethemen");
        ecke.setBackground(Color.GRAY.darker());
        ecke.setForeground(Color.WHITE);
        ecke.setOpaque(true);
        ecke.setBorder(BorderFactory.createLineBorder(Color.LIGHT_GRAY, 2));
        return ecke;
    }

    /**
     * @return Zeilenkpfe.
     */
    private JPanel erzeugeZeilenkoepfe() {
        JPanel panel = new JPanel();
        panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
        for (Kategorie kategorie : Kategorie.values()) {
            Zeilenkopf zeilenkopf;
            switch (kategorie) {
            case GESAMT:
            case OSUMME:
            case USUMME:
                zeilenkopf = new ZeilenkopfZwischen();
                break;
            case ESUMME:
                zeilenkopf = new ZeilenkopfEnde();
                break;
            default:
                zeilenkopf = new Zeilenkopf();
            }
            zeilenkopf.setText(kategorie.bezeichnung());
            zeilenkopf.setToolTipText(kategorie.beschreibung());
            zeilenkopf.setFont(zeilenkopf.getFont().deriveFont(Font.BOLD));
            zeilenkopf.setOpaque(true);
            panel.add(zeilenkopf);
        }
        return panel;
    }

    /**
     * Setzt Icons der oberen Eintrge.
     */
    private void setzeIcons() {
        for (Kategorie kategorie : Kategorie.OBEN) {
            int ind = kategorie.ordinal();
            Zeilenkopf zeilenkopf = (Zeilenkopf) zeilenkoepfe.getComponent(ind);
            zeilenkopf.setHorizontalTextPosition(SwingConstants.LEADING);
            zeilenkopf.setIcon(WUERFEL[ind]);
        }
    }

    /**
     * Sorgt fr die Synchronisation der Zeilenhhen der Punktetabellenzeilen mit der Tabellengre.
     */
    private void synchronisiereZeilenhoehen() {
        getViewport().addContainerListener(new ContainerAdapter() {
            @Override
            public void componentAdded(ContainerEvent event) {
                aktualisiereTabellengroesse();
                aktualisiereZeilenhoehe();
            }
        });
        getViewport().addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent event) {
                aktualisiereZeilenhoehe();
            }
        });
    }
}
