/******************************************************************************
 ** $Id: Kombinatorik.java 2533 2021-01-03 13:53:57Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ** Lauffhig ab Java 7.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.logik.analyse;

import jafuffy.logik.Turnier;

/** Sammlung von allgemeinen Formeln zum Thema Kombinatorik. */
public final class Kombinatorik {

    /** Anzahl Mglichkeiten, wenn alle Wrfel geworfen werden. */
    public static final int VARIATIONEN = 7776;
    /** Wahrscheinlichkeit bei einem Wrfel bei einem Wurfergebnis eine bestimmte Augenzahl auszulassen. */
    public static final double AUSLASSUNGSWAHRSCHEINLICHKEIT = 5. / 6.;

    /** Mittlere Augenzahl bei 0, 1, 2 oder 3 Restwrfen bei einem Wrfel. */
    private static final double[] AUGENZAHLEN = { 0, 3.5, 4.25, 14. / 3. };
    /**
     * Feld enthlt Wahrscheinlichkeiten fr k aus {0, 1, ..., 5} gleiche Augenzahlen ("k-Mehrling") in einem Wurfergebnis
     * ("Hieb") mit allen Wrfeln zu erzielen.
     */
    private static final double[] HIEBGLEICHENWAHRSCHEINLICHKEIT = {
        0,
        5. / 54.,
        25. / 36.,
        125. / 648.,
        25. / 1296.,
        1. / 1296. };
    /**
     * Feld enthlt die Wahrscheinlichkeiten, dass die aufsummierte Lnge aller Mehrlinge bei f freien oberen Eintrgen
     * m oder mehr betrgt in drei Wrfen mit allen Wrfeln bei f aus {1, 2, 3, 4, 5, 6} freien oberen Eintrgen und
     * Lngen m mit 0<=m<=5f.
     */
    private static final float[][] MEHRLINGSKUMULATIONSWAHRSCHEINLICHKEITEN = {
        // f = 1
        { 1.0000000f, 0.9350945f, 0.6988386f, 0.3548500f, 0.1044263f, 0.0132721f },
        // f = 2
        {
            1.0000000f,
            0.9998518f,
            0.9920864f,
            0.9405590f,
            0.7902726f,
            0.5417243f,
            0.2833378f,
            0.1069301f,
            0.0272795f,
            0.0042049f,
            0.0002956f },
        // f = 3
        {
            1.0000000f,
            1.0000000f,
            0.9999928f,
            0.9995734f,
            0.9943353f,
            0.9659221f,
            0.8793388f,
            0.7107106f,
            0.4851430f,
            0.2691175f,
            0.1176685f,
            0.0393608f,
            0.0096959f,
            0.0016566f,
            0.0001754f,
            0.0000087f },
        // f = 4
        {
            1.0000000f,
            1.0000000f,
            1.0000000f,
            0.9999999f,
            0.9999901f,
            0.9997585f,
            0.9973869f,
            0.9844010f,
            0.9404578f,
            0.8396740f,
            0.6734907f,
            0.4687962f,
            0.2757569f,
            0.1342929f,
            0.0531735f,
            0.0167854f,
            0.0041185f,
            0.0007567f,
            0.0000980f,
            0.0000080f,
            0.0000003f },
        // f = 5
        {
            1.0000000f,
            1.0000000f,
            1.0000000f,
            1.0000000f,
            1.0000000f,
            0.9999999f,
            0.9999958f,
            0.9999126f,
            0.9990586f,
            0.9939964f,
            0.9745622f,
            0.9223102f,
            0.8185786f,
            0.6609259f,
            0.4728535f,
            0.2937102f,
            0.1559390f,
            0.0698569f,
            0.0260761f,
            0.0079963f,
            0.0019781f,
            0.0003849f,
            0.0000567f,
            0.0000060f,
            0.0000004f,
            0.0000000f },
        // f = 6
        {
            1.0000000f,
            1.0000000f,
            1.0000000f,
            1.0000000f,
            1.0000000f,
            1.0000000f,
            1.0000000f,
            1.0000000f,
            0.9999988f,
            0.9999750f,
            0.9997136f,
            0.9979953f,
            0.9904755f,
            0.9669456f,
            0.9116353f,
            0.8105129f,
            0.6631048f,
            0.4886861f,
            0.3190120f,
            0.1821067f,
            0.0899730f,
            0.0381218f,
            0.0137222f,
            0.0041509f,
            0.0010409f,
            0.0002124f,
            0.0000344f,
            0.0000042f,
            0.0000004f,
            0.0000000f,
            0.0000000f } };
    /** Stirling-Zahlen der zweiten Art. S(nf,k) = k * S(n-1f,k) + S(n-1f,k-1); S(nf,0) = 0f, S(nf,1) = 1 */
    private static final int[][] STIRLING2;

    static {
        STIRLING2 = new int[Turnier.WUERFELSATZGROESSE + 1][];
        STIRLING2[0] = new int[1];
        STIRLING2[0][0] = 1;
        for (int n = 1; n <= Turnier.WUERFELSATZGROESSE; n++) {
            STIRLING2[n] = new int[n + 1];
            STIRLING2[n][0] = 0;
            for (int k = 1; k < n; k++) {
                STIRLING2[n][k] = k * STIRLING2[n - 1][k] + STIRLING2[n - 1][k - 1];
            }
            STIRLING2[n][n] = 1;
        }
    }

    /**
     * @param r
     *            Restliche Anzahl der freien Wrfe.
     * @return Mittlere erzielbare Augenzahl des Wrfels.
     */
    public static double augenzahl(int r) {
        return AUGENZAHLEN[r];
    }

    /**
     * Berechnet mittlere Augenzahl bei vorliegender Augenzahl und allen mglichen Restwrfen.
     * 
     * @param a
     *            Vorliegende Augenzahl des Wrfels.
     * @param r
     *            Restliche Anzahl r aus {0, 1, 2, 3} der freien Wrfe fr den Wrfel.
     * @return Mittlere erzielbare Augenzahl des Wrfels unter Bercksichtigung der vorliegenden Augenzahl.
     */
    public static double augenzahl(int a, int r) {
        return Double.max(a, augenzahl(r));
    }

    /**
     * @param n
     *            Nichtnegative Zahl.
     * @param k
     *            Ganze Zahl.
     * @return Binomialkoeffizient n ber k.
     */
    public static int binom(int n, int k) {
        if (k < 0 || k > n) {
            return 0;
        }
        if (2 * k > n) {
            return binom(n, n - k);
        }
        int koeffizient = 1;
        n -= k;
        for (int i = 1; i <= k; i++) {
            n++;
            koeffizient *= n;
            koeffizient /= i;
        }
        return koeffizient;
    }

    /**
     * @param n
     *            Nichtnegative Zahl
     * @return Fakultt
     */
    public static int fakultaet(int n) {
        int p = 1;
        while (n > 0) {
            p *= n;
            n--;
        }
        return p;
    }

    /**
     * @param n
     *            Startfaktor.
     * @param k
     *            Anzahl der Faktoren.
     * @return Fallende Fakultt.
     */
    public static int fallend(int n, int k) {
        int p;
        if (k < 0) {
            p = 0;
        } else {
            p = 1;
            for (int i = 0; i < k; i++, n--) {
                p *= n;
            }
        }
        return p;
    }

    /**
     * Ermittelt die mittlere Lnge eines Mehrlings zu einer fix vorgegebenen Augenzahl fr alle freien Wrfel.
     * 
     * @param n
     *            Anzahl der freien Wrfel.
     * @param r
     *            Anzahl der Restwrfe.
     * @return Mittlere Anzahl von Wrfeln mit gleicher Augenzahl mit einer fest vorgegebenen Augenzahl.
     */
    public static double fixgleichenmittel(int n, int r) {
        return n * fixgleichenwahrscheinlichkeit(1, r);
    }

    /**
     * Ermittelt die Wahrscheinlichkeit, dass alle Wrfel eines Teilwurfs einen Mehrling bilden zu einer fix
     * vorgegebenen Augenzahl bei r Restwrfen.
     * 
     * @param n
     *            Anzahl der freien Wrfel.
     * @param r
     *            Anzahl der Restwrfe (0 ist erlaubt).
     * @return Wahrscheinlichkeit fr genau n gleiche Wrfel bei festgelegter Augenzahl.
     */
    public static double fixgleichenwahrscheinlichkeit(int n, int r) {
        return fixgleichenwahrscheinlichkeit(n, n, r);
    }

    /**
     * Berechnet Wahrscheinlichkeit fr genau k Wrfel mit gleicher Augenzahl (k-Mehrling) nach r-maligem Wurfergebnis von n
     * Wrfeln bei vorab gegebener Augenzahl.
     * 
     * @param n
     *            Anzahl der freien Wrfel.
     * @param m
     *            Anzahl der gleichen Wrfel bei festgelegter Augenzahl (fixer m-Mehrling).
     * @param r
     *            Anzahl der Restwrfe.
     * @return Wahrscheinlichkeit fr einen fixen Mehrling mit vorab gegebener Augenzahl mit gegebener Lnge.
     */
    public static double fixgleichenwahrscheinlichkeit(int n, int m, int r) {
        double p = 1 - Math.pow(AUSLASSUNGSWAHRSCHEINLICHKEIT, r);
        return binom(n, m) * Math.pow(p, m) * Math.pow(1 - p, n - m);
    }

    /**
     * Berechnet die Wahrscheinlichkeit mit einem Wurfergebnis genau m Wrfel mit gleicher Augenzahl zu erhalten, wobei die
     * Augenzahl beliebig ist.
     * 
     * @param m
     *            Genaue Anzahl der Wrfel mit gleicher Augenzahl (m-Mehrling).
     * @return Wahrscheinlichkeit.
     */
    public static double gleichenwahrscheinlichkeit(int m) {
        return HIEBGLEICHENWAHRSCHEINLICHKEIT[m];
    }

    /**
     * Berechnet Wahrscheinlichkeit, mit der aufsummierten Lnge aller gnstigen Mehrlinge einen geforderten Mindestwert
     * zu bertreffen.
     * 
     * @param f
     *            Anzahl der freien oberen Eintrgen (0, 1, 2, 3, 4, 5, 6).
     * @param m
     *            Bentigte Mindestanzahl m>=0 der aufsummierten Lnge der Mehrlinge fr alle freien oberen Eintrge.
     * @return Wahrscheinlichkeit der Kumulation der Mehrlinge
     */
    public static float mehrlingskumulationswahrscheinlichkeit(int f, int m) {
        if (f == 0 && m == 0) {
            return 1;
        }
        if (f == 0 || m > Turnier.WUERFELSATZGROESSE * f) {
            return 0;
        }
        return MEHRLINGSKUMULATIONSWAHRSCHEINLICHKEITEN[f - 1][m];
    }

    /**
     * @param a
     *            Vorgegebene maximale Augenzahl, mit der ein m-Mehrling auftauchen soll.
     * @param m
     *            Genaue Lnge des Mehrlings mit der angegebenen Augenzahl. Im Wurfergebnis tritt kein lngerer Mehrling auf.
     * @return Wahrscheinlichkeit, den m-Mehrling (Einling, Zwilling, Drilling, Vierling, Fnfling) mit der Augenzahl a
     *         bei fnf Wrfeln in einem Wurfergebnis zur erzielen, wobei im Wurfergebnis kein gleich langer Mehrling zu einer greren
     *         Augenzahl auftritt.
     */
    public static double mehrlingswahrscheinlichkeit(int a, int m) {
        if (m == 2) {
            return (25. + 5 * (a - 1)) / 324.;
        } else {
            return gleichenwahrscheinlichkeit(m) / 6;
        }
    }

    /**
     * @param min
     *            Mindestanzahl der Wrfel, welche eine bestimmte Augenzahl zeigen sollen.
     * @return Wahrscheinlichkeit fr die Mindestanzahl.
     */
    public static double mindestfixgleichenwahrscheinlichkeit(int min) {
        return mindestfixgleichenwahrscheinlichkeit(Turnier.WUERFELSATZGROESSE, min, Turnier.WURFVERSUCHANZAHL);
    }

    /**
     * @param n
     *            Anzahl der freien Wrfel.
     * @param min
     *            Mindestanzahl der Wrfel, welche eine bestimmte Augenzahl zeigen sollen.
     * @param r
     *            Anzahl der verbleibenden Restwrfe.
     * @return Wahrscheinlichkeit fr die Mindestanzahl.
     */
    public static double mindestfixgleichenwahrscheinlichkeit(int n, int min, int r) {
        double p = 0;
        for (int k = min; k <= Turnier.WUERFELSATZGROESSE; k++) {
            p += fixgleichenwahrscheinlichkeit(n, k, r);
        }
        return p;
    }

    /**
     * @param min
     *            Mindestanzahl der Wrfel mit gleicher Augenzahl (1, 2, 3, 4, 5).
     * @return Wahrscheinlichkeit fr mindestens m Wrfel mit gleicher Augenzahl, wobei die Augenzahl beliebig ist. Im
     *         Wurfergebnis werden die Wrfel mit gleicher Augenzahl gezhlt; es gilt die lngste Sequenz, wobei bei gleich
     *         langen Sequenzen diejenige zu der hheren Augenzahl gewhlt wird.
     */
    public static double mindestgleichenwahrscheinlichkeit(int min) {
        double p = 0;
        for (int k = min; k <= Turnier.WUERFELSATZGROESSE; k++) {
            p += gleichenwahrscheinlichkeit(k);
        }
        return p;
    }

    /**
     * @param a
     *            Vorgegebene maximale Augenzahl, mit der mindestens ein k-Mehrling auftauchen soll.
     * @param m
     *            Minimale Lnge eines Mehrlings mit der angegebenen Augenzahl.
     * @return Wahrscheinlichkeit, mindestens einen m-Mehrling (Einling, Zwilling, Drilling, Vierling, Fnfling) mit der
     *         gegebenen Augenzahl bei fnf Wrfeln in einem Wurfergebnis zur erzielen, wobei kein gleich langer Mehrling mit
     *         einer greren Augenzahl auftritt.
     */
    public static double mindestmehrlingswahrscheinlichkeit(int a, int m) {
        double p = 0;
        for (int k = m; k <= Turnier.WUERFELSATZGROESSE; k++) {
            p += mehrlingswahrscheinlichkeit(a, k);
        }
        return p;
    }

    /**
     * Berechnet Stirlingzahl der zweiten Art (Anzahl der Partitionen der Menge der Kardinalitt n in k nicht-leere
     * Mengen).
     *
     * @param n
     *            Kardinalitt der Menge.
     * @param k
     *            Anzahl der nicht-leeren Mengen.
     * @return Stirlingzahl der zweiten Art.
     */
    public static int stirling2(int n, int k) {
        if (k <= n) {
            return STIRLING2[n][k];
        } else {
            return 0;
        }
    }

    /**
     * Berechnet die Anzahl aller mglichen Positionen fr einen Wrfelsatz, auf denen eine Straenteilsequenz fester
     * Lnge verteilt werden kann.
     * 
     * @param n
     *            Lnge des vorgegebenen Wrfelsatzes.
     * @param d
     *            Anzahl der Wrfel, die nicht zur Straenteilschablone beitragen.
     * @param k
     *            Lnge der Straenteilsequenz.
     * @return Anzahl der mglichen Positionen.
     */
    public static int strassenteilschablone(int n, int d, int k) {
        int summe = 0;
        for (int j = k; j <= n; j++) {
            summe += stirling2(j, k) * binom(n, j) * Math.pow(d, n - j);
        }
        return summe;
    }

    /**
     * Berechnet die Anzahl aller Mglichkeiten, aus einem Wrfelsatz eine Straenteilsequenz mit gewnschter Lnge zu
     * erwrfeln, wobei die Teilsequenz nicht den ganzen Wrfelsatz umfassen muss.
     * 
     * @param n
     *            Lnge des vorgegebenen Wrfelsatzes.
     * @param d
     *            Anzahl der Wrfel, die nicht zur Straenteilsequenz beitragen.
     * @param k
     *            Lnge der Straenteilsequenz.
     * @param m
     *            Die Anzahl der gnstigen Augenzahlen, welche auf die Straenteilsequenz verteilt werden drfen.
     * @return Anzahl der mglichen Wrfelkombinationen
     */
    public static double strassenteilsequenz(int n, int d, int k, int m) {
        return fallend(m, k) * strassenteilschablone(n, d, k);
    }

    /** Diese Klasse liefert nur statische Methode, daher wird eine Instanziierung verboten. */
    private Kombinatorik() {
    }

}
