/******************************************************************************
 ** $Id: Anmeldung.java 2514 2020-09-13 19:21:59Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.netzwerk;

import java.io.Serializable;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.LinkedHashSet;

/** Anmeldungsnachricht. */
class Anmeldung implements Serializable {

    private static final long serialVersionUID = -2864254703793642525L;

    /** Der Hashwert ber alle Teilnehmer an allen Austragungssttten, von dieser Austragungssttte berechnet. */
    private final byte[] teilnehmerhashwert;
    /** Die Identifikation des Vorsitzes. */
    private final Kontakt vorsitzkontakt;
    /** Der Zeitpunkt, zu dem die Meldung zum Vorsitz des Turniers gemeldet wurde. */
    private final LocalDateTime vorsitzmeldungszeit;
    /** Kontaktdaten des Absenders dieser Nachricht. */
    protected final Kontakt vorortkontakt;
    /** Auflistung aller Teilnehmer, wie bei der Absprache gewhlt und auf dem Zettel angezeigt. */
    protected final LinkedHashSet<String> vorortteilnehmer;

    /**
     * Erzeugung einer Anmeldungsnachricht.
     *
     * @param kontakt
     *            Kontaktdaten des Absenders dieser Anmeldung.
     * @param vorortteilnehmer
     *            Teilnehmer an der Austragungssttte des Absenders; darf null sein.
     */
    protected Anmeldung(Kontakt kontakt, LinkedHashSet<String> vorortteilnehmer) {
        this(kontakt, vorortteilnehmer, null, null, null);
    }

    /**
     * Erstellt eine Anmeldung.
     *
     * @param vorortkontakt
     *            Kontaktdaten des Absenders.
     * @param vorortteilnehmer
     *            Teilnehmer an der Austragungssttte des Absenders; darf null sein.
     * @param teilnehmerhashwert
     *            Der Hashwert ber alle Teilnehmer an allen Austragungssttten von dieser Austragungssttte berechnet.
     * @param vorsitzkontakt
     *            Die Identifikation der Austragungssttte, welche den Vorsitz bernehmen mchte.
     * @param vorsitzmeldungszeit
     *            Der Zeitpunkt, zu dem die Meldung zum Vorsitz des Turniers gemeldet wurde; darf null sein.
     */
    Anmeldung(Kontakt vorortkontakt, LinkedHashSet<String> vorortteilnehmer, byte[] teilnehmerhashwert,
            Kontakt vorsitzkontakt, LocalDateTime vorsitzmeldungszeit) {
        this.vorortkontakt = vorortkontakt;
        this.vorortteilnehmer = vorortteilnehmer;
        this.teilnehmerhashwert = teilnehmerhashwert;
        this.vorsitzkontakt = vorsitzkontakt;
        this.vorsitzmeldungszeit = vorsitzmeldungszeit;
    }

    @Override
    public String toString() {
        return vorsitzmeldungszeit + "@" + vorortkontakt + "|" + vorsitzkontakt + " # "
                + Arrays.toString(teilnehmerhashwert);
    }

    /** Gibt den hier errechneten Hashwert bezglich aller Teilnehmer zurck. */
    byte[] teilnehmerhashwert() {
        return teilnehmerhashwert;
    }

    /** Liefert Absenderadresse. */
    Kontakt vorortkontakt() {
        return vorortkontakt;
    }

    /** Liefert die Teilnehmer des Turniers der Austragungssttte, welche diese Nachricht verschickt hat. */
    LinkedHashSet<String> vorortteilnehmer() {
        return vorortteilnehmer;
    }

    /** Liefert die Identifikation des Vorsitzes. */
    Kontakt vorsitzkontakt() {
        return vorsitzkontakt;
    }

    /** Liefert den Zeitpunkt der Meldung zum Vorsitz. */
    LocalDateTime vorsitzmeldungszeit() {
        return vorsitzmeldungszeit;
    }

}