/******************************************************************************
 ** $Id: Leitung.java 2610 2021-02-01 01:16:44Z wmh $
 ** Diese Datei ist Bestandteil der Java-Quelltexte des Wrfelspiels JaFuffy.
 ******************************************************************************
 ** Copyright (C) Wolfgang Hauck <wolfgang.hauck@3kelvin.de>
 ******************************************************************************
 ** This program is free software: you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, either version 3 of the License, or
 ** (at your option) any later version.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************
 ** Die aktuellste Version von JaFuffy findet sich im Internet unter
 ** <http://jafuffy.3kelvin.de>.
 **
 ** Kommentare, Fehler oder Erweiterungswnsche bitte per E-Mail senden an
 ** <jafuffy@3kelvin.de>.
 ******************************************************************************/
package jafuffy.netzwerk;

import java.awt.Component;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.TreeMap;

import javax.swing.JCheckBox;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;

import jafuffy.logik.Manager;
import jafuffy.logik.Plan;
import jafuffy.logik.Spieler;

/** bernimmt die Leitung der Organisation eines Turniers. */
public class Leitung extends AbstractTableModel implements ActionListener, PropertyChangeListener {

    /** Schnittstelle zur Aufnahme des Turnierbetriebs nach erfolgter Organisation bers Netzwerk. */
    public interface Aufnahme {
        /** Nimmt in der Absprache Anpassungen an, welche sich aus der Aufnahme des Turnierbetriebs ergeben. */
        void folge();
    }

    /** Zur Darstellung von Eigenschaften im Organisationsdialog, die sich verndern. */
    public interface Darstellung {
        /** Aktualisiert einige Darstellungselemente des Organisationsdialoges. */
        void aktualisiere();

        /** Folgt im Organisationsdialog der Aufnahme eines Turniers im Netzwerkbetrieb. */
        void folge();

        /**
         * Anzeige der Bestandteile einer Chat-Nachricht im Organisationsdialog.
         *
         * @param kontakt
         * @param teilnehmer
         * @param text
         */
        void zeige(Kontakt kontakt, String teilnehmer, String text);
    }

    /** Die Phase fr die Durchfhrung eines Turniers ber ein Netzwerk. */
    public enum Phase {
        /** Die Bedingungen fr das Turnier werden abgestimmt. */
        ABSTIMMUNG,
        /** Herstellung der Verbindungen zwischen allen Austragungsorten. */
        VERBINDUNG,
        /** Der Turnierbetrieb ber ein Netzwerk ist mglich, alle Verbindungen stehen. */
        BEREIT,
        /** Der Turnierbetrieb ist am Laufen. */
        BETRIEB,
        /** Der Turnierbetrieb wird abgebrochen. */
        ABBRUCH
    }

    /** Die Spalten der Tabelle der Anmeldungen ber das lokale Netzwerk. */
    private enum Spalte {
        TEILNEHMER("Teilnehmer", 250), IDENTIFIKATION("Identifikation", 250);

        /** Spaltentitel fr Anzeige in Tabelle. */
        private final String titel;
        /** Spaltenbreite fr Anzeige in Tabelle. */
        final int breite;

        /**
         * Erstellt eine Aufzhlungselement.
         *
         * @param titel
         *            Spaltentitel.
         * @param breite
         *            Spaltenbreite.
         */
        Spalte(String titel, int breite) {
            this.titel = titel;
            this.breite = breite;
        }

        @Override
        public String toString() {
            return titel;
        }
    }

    /** Zur berprfung, ob die Anmeldung eines Austragungsortes noch aktuell ist. */
    private class Ueberpruefung extends TimerTask {
        /** Periode des Anstoes zur berprfung. */
        private static final int UEBERWACHUNGSPERIODE = 2000;

        /** Identifikation der Austragungssttte, die berprft wird. */
        private final Kontakt kontakt;
        /** Wird bei jeder Anmeldungen auf wahr gesetzt, und regelmig auf falsch gesetzt. */
        private boolean vorhanden;
        /** bernimmt die regelmige berwachungsaktion. */
        private final Timer ueberwachung;

        /**
         * Erstellt die berprfung einer Austragungssttte.
         *
         * @param kontakt
         *            Identifikation der Austragungssttte
         */
        Ueberpruefung(Kontakt kontakt) {
            this.kontakt = kontakt;
            ueberpruefungen.put(kontakt, this);
            ueberwachung = new Timer(Ueberpruefung.class.getSimpleName());
            ueberwachung.schedule(this, UEBERWACHUNGSPERIODE, UEBERWACHUNGSPERIODE);
        }

        @Override
        synchronized public void run() {
            if (!vorhanden) {
                ueberwachung.cancel();
                deregistriere(kontakt);
                fireTableDataChanged();
            }
            vorhanden = false;
        }

        /** Besttigt, dass die Anmeldung einer Austragungssttte noch aktuell ist. */
        synchronized void bestaetige() {
            vorhanden = true;
        }
    }

    /** Beschreibt, wer den Vorsitz in der Organisation des Turniers hat. */
    private class Vorsitz {
        /** Der Kontakt des Vorsitzes. */
        private Kontakt kontakt;
        /** Die Zeit, zu welcher der Kontakt des Vorsitzes gemeldet wurde. */
        private LocalDateTime meldungszeit;

        /** Gibt an, ob der hiesige Kontakt gleich dem Kontakt des Vorsitzes ist. */
        boolean ausfuehrend() {
            return ausfuehrend(vorortkontakt);
        }

        /**
         * Gibt an, ob ein Kontakt gleich dem Kontakt des Vorsitzes ist.
         *
         * @param kontakt
         *            Der angefragte Kontakt.
         * @return
         */
        boolean ausfuehrend(Kontakt kontakt) {
            return kontakt != null && kontakt.equals(this.kontakt);
        }

        /**
         * Gibt an, die Meldungszeit nach der Meldungszeit des derzeitigen Vorsitzes liegt.
         *
         * @param meldungszeit
         *            Die angefragte Meldungszeit.
         * @return Wahr, falls die Meldungszeit neuer ist; falsch ansonsten.
         */
        boolean danach(LocalDateTime meldungszeit) {
            return meldungszeit != null && (this.meldungszeit == null || meldungszeit.isAfter(this.meldungszeit));
        }

        /**
         * Speichert den Kontakt des Vorsitzes.
         *
         * @param kontakt
         *            Der zu setzende Kontakt.
         * @param meldungszeit
         *            Die Zeit, zu der der Kontakt gemeldet wurde.
         */
        void setze(Kontakt kontakt, LocalDateTime meldungszeit) {
            this.kontakt = kontakt;
            this.meldungszeit = meldungszeit;
            pruefe();
        }

        /** Gibt an, ob der Vorsitz schon vergeben ist. */
        boolean vakant() {
            return kontakt == null;
        }
    }

    private static final long serialVersionUID = -3855732214488995777L;

    /** Periode in Millisekunden, mit der versucht wird, Verbindungen aufzusetzen.s */
    private static final int VERBINDUNGSPERIODE = 1000;
    /** Anzahl der Verbindungsversuche, bevor aufgegeben wird. */
    private static final int VERBINDUNGSVERSUCHE = 3;

    /** Port-Nummer zwecks Organisation eines Turniers ber ein Netzwerk. */
    static final int PORT = 54815;
    /** IP-Adresse zwecks Organisation eines Turniers ber ein Netzwerk. */
    static final String ADRESSE = "239.255.255.136";

    /** Meldet einen Austragungssttte zur Teilnahme an. */
    private final Anmelder anmelder;
    /** Ermglicht die Einschreibung in ein ausgeschriebenes Turnier. */
    private final Annehmer annehmer;
    /** Zur Herstellung aller Verbindungen und deren Betreuung. */
    private final Vermittlung vermittlung;
    /** Die vollstndige Absenderadresse der Austragungsorte, an dessen Gert die Teilnehmer sitzen. */
    private final Kontakt vorortkontakt;
    /** Alle gefundenen Kandidaten der Austragungsorte, welche ber ein Netzwerk am Turnier teilnehmen mchten. */
    private final TreeMap<Kontakt, LinkedHashSet<String>> namengruppen = new TreeMap<>();
    /** berwachung-Threads fr alle Kandidaten, ob diese whrend Abstimmung noch vorhanden ist. */
    private final HashMap<Kontakt, Ueberpruefung> ueberpruefungen = new HashMap<>();
    /** Zur Berechnung von Hashwerten. */
    private final MessageDigest md;
    {
        MessageDigest instanz = null;
        try {
            instanz = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException ereignis) {
            ereignis.printStackTrace();
        }
        md = instanz;
    }
    /** Die den Austragungssttten bekannten Hashwerte ber alle Turnierteilnehmer. */
    private final HashMap<Kontakt, byte[]> teilnehmerhashwerte = new HashMap<>();
    /** Beschreibt, wer aus Sicht der Austragungssttte den Vorsitz fhrt. */
    private final Vorsitz vorsitz = new Vorsitz();

    /** Alle Teilnehmer aus allen Austragungsorten in einer einzigen Gesamtliste zusammengestellt. */
    private LinkedHashSet<String> gesamtnamen;
    /** Zur Delegation der Anzeige von Nachrichten an das zustndige Element der Benutzeroberflche. */
    private Darstellung darstellung;
    /** Gibt an, ob die hiesige Austragungssttte die Bereitschaft zur Durchfhrung eines Turniers sieht. */
    private boolean konsistent;
    /** Die derzeitig aktive Phase der Leitung. */
    private volatile Phase phase = Phase.ABSTIMMUNG;
    /** Identifikationen der Austragungssttten, fr die eine Anmeldung gefunden wurde und laufend besttigt wird. */
    private Kontakt[] kontakte = new Kontakt[Spieler.SPIELER];

    /**
     * Turnierleitung nimmt Arbeit auf.
     *
     * @param aufnahme
     *            Schnittstelle zur Anpassung der Absprache nach Aufnahme des Turnierbetriebs.
     * @param manager
     *            Der Manager, welcher Turnier im Allgemeinen betreut.
     * @param vorortnamen
     *            Die Teilnehmer des Turniers der Austragungssttte, an dessen Gert die Teilnehmer direkt sitzen.
     * @throws UnknownHostException
     *             Geworfen bei Kommunikationsproblemen ber das Netzwerk.
     * @throws SocketException
     *             Geworfen bei Kommunikationsproblemen ber das Netzwerk.
     */
    public Leitung(Aufnahme aufnahme, Manager manager, LinkedHashSet<String> vorortnamen)
            throws UnknownHostException, SocketException {
        vermittlung = manager.vermittlung(plan -> {
            SwingUtilities.invokeLater(() -> {
                aufnahme.folge();
                plan.adaptiere(vorortnamen);
                betreibe(manager, plan);
                phase = Phase.BETRIEB;
            });
        });
        vorortkontakt = new Kontakt(InetAddress.getLocalHost(), vermittlung.socket().getLocalPort());
        annehmer = new Annehmer();
        anmelder = new Anmelder(vorortkontakt, vorortnamen);
        annehmer.addPropertyChangeListener(ereignis -> {
            String eigenschaft = ereignis.getPropertyName();
            Object wert = ereignis.getNewValue();
            if (eigenschaft.equals(Anmeldung.class.getSimpleName())) {
                verarbeite((Anmeldung) wert);
            } else if (eigenschaft.equals(Nachricht.class.getSimpleName())) {
                verarbeite((Nachricht) wert);
            } else if (eigenschaft.equals(Bescheid.class.getSimpleName())) {
                verarbeite((Bescheid) wert);
            }
        });
    }

    @Override
    public void actionPerformed(ActionEvent ereignis) {
        String kommando = ereignis.getActionCommand();
        switch (kommando) {
        case "Aufnahme":
            initiiere(SwingUtilities.getWindowAncestor((Component) ereignis.getSource()));
            break;
        case "Abbruch":
            stoppe(SwingUtilities.getWindowAncestor((Component) ereignis.getSource()));
            break;
        case "Vorsitz":
            bewerbe(((JCheckBox) ereignis.getSource()).isSelected());
            break;
        default:
        }
    }

    /** Gibt an, ob die hiesige Austragungssttte derzeit den Vorsitz innehat. */
    public boolean ausfuehrend() {
        synchronized (anmelder) {
            return vorsitz.ausfuehrend();
        }
    }

    /**
     * bergibt den Betrieb eines Turniers ber ein Netzwerk nach einem Plan an den zustndigen Manager.
     *
     * @param manager
     *            Der zustndige Manager.
     * @param plan
     *            Der Plan fr das Turnier.
     */
    public void betreibe(Manager manager, Plan plan) {
        manager.betreibe(plan, vermittlung);
    }

    /** Liefert Spaltenbreite fr Anzeige in einer Tabelle. */
    public int breite(int spalte) {
        return Spalte.values()[spalte].breite;
    }

    /**
     * Liefert alle Teilnehmer aus allen Austragungsorten in einer einzigen Gesamtliste zusammengestellt.
     *
     * @return Eine Menge aus Namen, welche die Reihenfolgen aus Anmeldungen respektiert.
     */
    public LinkedHashSet<String> gesamtnamen() {
        return gesamtnamen;
    }

    @Override
    public int getColumnCount() {
        return Spalte.values().length;
    }

    @Override
    public String getColumnName(int spalte) {
        return Spalte.values()[spalte].toString();
    }

    @Override
    public int getRowCount() {
        return namengruppen.size();
    }

    @Override
    public Object getValueAt(int zeile, int spalte) {
        String identifikation = kontakte[zeile].toString();
        switch (Spalte.values()[spalte]) {
        case IDENTIFIKATION:
            return identifikation;
        case TEILNEHMER:
            return namengruppen.get(kontakte[zeile]);
        default:
            return null;
        }
    }

    /**
     * Gibt an, ob die angegebene Tabellenzeile der hiesigen Austragungssttte entspricht.
     *
     * @param zeile
     *            Die Zeilennummer der Tabelle.
     */
    public boolean heimisch(int zeile) {
        return vorortkontakt.equals(kontakte[zeile]);
    }

    /**
     * Informiert alle Teilnehmer an einem Turnier ber ein Netzwerk ber den ausgehandelten Plan.
     *
     * @param plan
     *            Der ausgehandelte Plan
     * @throws Exception
     *             Ein Problem ist bei der Verteilung der Information aufgetreten.
     */
    public void informiere(Plan plan) throws Exception {
        vermittlung.verteile(plan);
        phase = Phase.BETRIEB;
    }

    /** Gibt an, ob alle Information ntig zum Turnierbetrieb ber ein Netzwerk konsistent vorhanden sind. */
    public boolean konsistent() {
        return konsistent;
    }

    /** Liefert die Phase in der Leitung eines Turniers im Netzwerkbetrieb zurck. */
    public Phase phase() {
        return phase;
    }

    @Override
    public void propertyChange(PropertyChangeEvent ereignis) {
        String eigenschaft = ereignis.getPropertyName();
        Object wert = ereignis.getNewValue();
        if (eigenschaft.equals("Senden")) {
            anmelder.verschicke(new Nachricht(vorortkontakt, vorortnamen(), wert.toString()));
        }
    }

    /** Stoppt die Leitung eines Turniers im Netzwerkbetrieb zwecks Einstellung des Turniers. */
    public void stoppe() {
        vermittlung.schliesse();
    }

    /**
     * Verbindet mit Darstellungsfunktionen der Benutzeroberflche.
     *
     * @param darstellung
     *            Eine Sammlung von Methode des Organisationsdialoges.
     */
    public void verknuepfe(Darstellung darstellung) {
        this.darstellung = darstellung;
        new Thread(annehmer).start();
        anmelder.starte();
    }

    /** Gibt alle Namen des Vorortkontaktes zurck. */
    public LinkedHashSet<String> vorortnamen() {
        return namengruppen.get(vorortkontakt);
    }

    /** Baut Informationen ber den Vorsitz zusammen, und zwar alle Namen mit Kontaktdaten, getrennt durch "@". */
    public String vorsitz() {
        return namengruppen.get(vorsitz.kontakt) + "@" + vorsitz.kontakt.toString();
    }

    /** Gibt an, ob der hiesige Austragungsort den Vorsitz der Turniervorbereitung ber ein Netzwerk hat. */
    public boolean vorsitzend() {
        synchronized (anmelder) {
            return vorsitz.ausfuehrend();
        }
    }

    /**
     * Gibt an, ob der Kontakt vertreten durch die Zeile aus der Organisationstabelle den Vorsitz hat.
     *
     * @param zeile
     *            Zeilenindex aus der Tabelle der Kontakte.
     * @return Wahr, falls Vorsitz; falsch andererseits.
     */
    public boolean vorsitzend(int zeile) {
        synchronized (anmelder) {
            return vorsitz.ausfuehrend(kontakte[zeile]);
        }
    }

    /**
     * Wartet darauf, dass alle Verbindungen zu allen Austragungsorten hergestellt wurde.
     *
     * @throws InterruptedException
     *             Nicht alle Verbindungen sind zustande gekommen.
     * @return Derzeitige Phase des Standes der Turnierablaufs ber ein Netzwerk.
     */
    public boolean warte() throws InterruptedException {
        boolean fertig = vermittlung.warte();
        if (fertig) {
            phase = Phase.BEREIT;
        } else {
            phase = Phase.ABBRUCH;
        }
        return fertig;
    }

    /** Beendet die Infrastruktur, welche die Abstimmung ber ein Netzwerk durchfhrt. */
    private void beendeAbstimmung() {
        synchronized (anmelder) {
            anmelder.stoppe();
            annehmer.stoppe();
        }
    }

    /**
     * Benennt den Turniervorsitz mit Aktualisierung der Datenhaltung in der Austragungssttte vor Ort.
     *
     * @param absenderkontakt
     *            Die Kontaktdaten des Absenders
     * @param meldungskontakt
     *            Der Kontakt, welcher die Meldung zur bernahme des Vorsitzes abgesetzt hat
     * @param meldungszeit
     *            Die Zeit, zu der die Meldung erfolgt ist
     */
    private void benenne(Kontakt absenderkontakt, Kontakt meldungskontakt, LocalDateTime meldungszeit) {
        if (vorsitz.danach(meldungszeit)) {
            if (meldungskontakt == null) {
                if (vorsitz.ausfuehrend(absenderkontakt)) {
                    vorsitz.setze(null, LocalDateTime.now());
                    anmelder.setze(vorsitz.kontakt, vorsitz.meldungszeit);
                }
            } else if (meldungskontakt.equals(absenderkontakt)) {
                vorsitz.setze(meldungskontakt, meldungszeit);
                anmelder.setze(vorsitz.kontakt, vorsitz.meldungszeit);
            }
            fireTableDataChanged();
        }
    }

    /**
     * Bewirbt sich um den Vorsitz bei der Turnierorganisation.
     *
     * @param bereit
     *            Gibt an, ob Bewerbung erfolgt ist, oder nicht.
     */
    private void bewerbe(boolean bereit) {
        synchronized (anmelder) {
            Kontakt kontakt = vorortkontakt;
            if (!bereit && (vorsitz.ausfuehrend() || vorsitz.vakant())) {
                kontakt = null;
            }
            anmelder.setze(kontakt, LocalDateTime.now());
        }
    }

    /**
     * Deregistriert einen Kontakt.
     *
     * @author kontakt Der zu deregistrierende Kontakt.
     */
    private void deregistriere(Kontakt kontakt) {
        synchronized (anmelder) {
            ueberpruefungen.remove(kontakt);
            namengruppen.remove(kontakt);
            kontakte = namengruppen.keySet().toArray(new Kontakt[0]);
            teilnehmerhashwerte.remove(kontakt);
            benenne(kontakt, null, LocalDateTime.now());
            hashe();
            pruefe();
        }
    }

    /** Setzt den aktualisierten Teilnehmerhash zwecks Versand ber das Netzwerk. */
    private void hashe() {
        try (ByteArrayOutputStream feldstrom = new ByteArrayOutputStream();
                ObjectOutputStream objektstrom = new ObjectOutputStream(feldstrom)) {
            for (Map.Entry<Kontakt, LinkedHashSet<String>> eintrag : namengruppen.entrySet()) {
                objektstrom.writeObject(eintrag.getKey());
                objektstrom.writeObject(eintrag.getValue());
            }
            byte[] vororthashwert = md.digest(feldstrom.toByteArray());
            teilnehmerhashwerte.put(vorortkontakt, vororthashwert);
            anmelder.setze(vororthashwert);
        } catch (IOException ausnahme) {
            ausnahme.printStackTrace();
        }
    }

    /** Initiiert die wechselseitigen Verbindungen zwischen allen Austragungsorten. */
    private void initiiere(Window fenster) {
        fenster.dispose();
        final Bescheid bescheid = new Bescheid();
        final Timer timer = new Timer(Leitung.class.getSimpleName());
        timer.schedule(new TimerTask() {
            /** Zhlt die Versuche, alle Verbindungen anzustoen. */
            private int anzahl;

            @Override
            public void run() {
                if (phase == Phase.ABSTIMMUNG && anzahl <= VERBINDUNGSVERSUCHE) {
                    anzahl++;
                    anmelder.verschicke(bescheid);
                } else {
                    cancel();
                    beendeAbstimmung();
                    if (phase == Phase.ABSTIMMUNG) {
                        vermittlung.schliesse();
                    }
                }
            }
        }, 0, VERBINDUNGSPERIODE);
    }

    /**
     * Prft, ob alle Informationen konsistent vorliegen, sodass der Turnierbetrieb aufgenommen werden kann; beinhaltet
     * die Aktualisierung der Menge aller bekannten Teilnehmernamen.
     */
    private void pruefe() {
        int spieleranzahl = 0;
        for (LinkedHashSet<String> namen : namengruppen.values()) {
            spieleranzahl += namen.size();
        }
        gesamtnamen = new LinkedHashSet<>(Spieler.SPIELER);
        for (LinkedHashSet<String> namen : namengruppen.values()) {
            gesamtnamen.addAll(namen);
        }
        konsistent = true;
        boolean vorhanden = false;
        byte[] vororthashwert = teilnehmerhashwerte.get(vorortkontakt);
        for (Kontakt kontakt : kontakte) {
            vorhanden = vorhanden || vorsitz.ausfuehrend(kontakt);
            konsistent = konsistent && Arrays.equals(vororthashwert, teilnehmerhashwerte.get(kontakt));
        }
        konsistent = spieleranzahl <= Spieler.SPIELER && spieleranzahl == gesamtnamen.size() && namengruppen.size() > 1
                && vorhanden && konsistent;
        darstellung.aktualisiere();
    }

    /**
     * Registriert eine Anmeldung.
     *
     * @param kontakt
     *            Kontaktdaten des Anmelders.
     * @param teilnehmerhashwert
     *            Der dem Anmelder bekannte Hashwert ber alle Teilnehmer.
     */
    private void registriere(Kontakt kontakt, byte[] teilnehmerhashwert) {
        teilnehmerhashwerte.put(kontakt, teilnehmerhashwert);
        pruefe();
    }

    /**
     * Registriert eine Anmeldung.
     *
     * @param kontakt
     *            Kontaktdaten des Anmelders.
     * @param vorortteilnehmer
     *            Teil an der Austragungssttte des Anmelders.
     * @param teilnehmerhashwert
     *            Der dem Anmelder bekannte Hashwert ber alle Teilnehmer.
     */
    private void registriere(Kontakt kontakt, LinkedHashSet<String> vorortteilnehmer, byte[] teilnehmerhashwert) {
        namengruppen.put(kontakt, vorortteilnehmer);
        kontakte = namengruppen.keySet().toArray(new Kontakt[0]);
        hashe();
        registriere(kontakt, teilnehmerhashwert);
    }

    /**
     * Stoppt die Organisation des Turnierbetriebs ber ein lokales Netzwerk.
     *
     * @param fenster
     *            Das Dialogfenster, welches zu diesem Modell gehrt.
     */
    private void stoppe(Window fenster) {
        phase = Phase.ABBRUCH;
        beendeAbstimmung();
        fenster.dispose();
    }

    /**
     * Auswertung einer Nachricht, die zwecks Turnieranmeldung bers Netz verschickt und hier vor Ort erhalten wurde.
     *
     * @param anmeldung
     *            Die zu bearbeitende Anmeldung.
     */
    private void verarbeite(Anmeldung anmeldung) {
        synchronized (anmelder) {
            Ueberpruefung ueberpruefung = ueberpruefungen.get(anmeldung.vorortkontakt());
            if (ueberpruefung == null) {
                registriere(anmeldung.vorortkontakt(), anmeldung.vorortteilnehmer(), anmeldung.teilnehmerhashwert());
                new Ueberpruefung(anmeldung.vorortkontakt());
                fireTableDataChanged();
            } else {
                ueberpruefung.bestaetige();
                registriere(anmeldung.vorortkontakt(), anmeldung.teilnehmerhashwert());
            }
            benenne(anmeldung.vorortkontakt(), anmeldung.vorsitzkontakt(), anmeldung.vorsitzmeldungszeit());
        }
    }

    /**
     * Auswertung des Bescheids, welcher die Turniervorbereitungen ber das Netzwerk abschliet, so dass nach Rckkehr
     * aus dieser Methode keine Abstimmung ber Teilnehmer mehr stattfindet, wechselseitig aber zwischen allen
     * Austragungsorten Verbindungen bestehen.
     *
     * @param bescheid
     *            Der zu bearbeitende Bescheid.
     */
    private void verarbeite(Bescheid bescheid) {
        if (phase == Phase.VERBINDUNG) {
            return;
        }
        phase = Phase.VERBINDUNG;
        beendeAbstimmung();
        for (Ueberpruefung ueberpruefung : ueberpruefungen.values()) {
            ueberpruefung.bestaetige();
            ueberpruefung.cancel();
        }
        ueberpruefungen.clear();
        vermittlung.verbinde(kontakte, vorortkontakt);
        if (!vorsitzend()) {
            darstellung.folge();
        }
    }

    /**
     * Sorgt fr die Anzeige einer Textnachricht.
     *
     * @param nachricht
     *            Die anzuzeigende Nachricht.
     */
    private void verarbeite(Nachricht nachricht) {
        synchronized (anmelder) {
            darstellung.zeige(nachricht.vorortkontakt(), nachricht.vorortteilnehmer().toString(), nachricht.text());
        }
    }

}
